<?php

declare(strict_types=1);

namespace Drupal\recurring_events_registration\Form;

use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Url;
use Drupal\recurring_events\Entity\EventInstance;
use Drupal\recurring_events_registration\Enum\RegistrationType;
use Drupal\recurring_events_registration\NotificationService;
use Drupal\recurring_events_registration\RegistrationCreationService;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Registrant contact form.
 */
class ContactForm extends FormBase {

  use AutowireTrait;

  /**
   * The event instance object.
   *
   * @var \Drupal\recurring_events\Entity\EventInstance|mixed
   */
  protected EventInstance $eventInstance;

  public function __construct(
    protected RegistrationCreationService $creationService,
    protected NotificationService $notificationService,
    protected MailManagerInterface $mail,
    protected LanguageManagerInterface $languageManager,
  ) {
    $request = $this->getRequest();
    $params = $request->attributes->all();
    if (!empty($params['eventinstance'])) {
      $event_instance = $params['eventinstance'];
      $this->eventInstance = $event_instance;
      $this->creationService->setEventInstance($event_instance);
    }
    else {
      throw new NotFoundHttpException();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'recurring_events_registration_contact_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $registered = $this->creationService->retrieveRegisteredParties(TRUE, FALSE, FALSE);
    $waitlisted = $this->creationService->retrieveWaitlistedParties();

    $form['header'] = [
      '#type' => 'markup',
      '#markup' => $this->t('By submitting this form you will be contacting %registered registrants and/or %waitlisted people on the waitlist for the %name @type.', [
        '%registered' => count($registered),
        '%waitlisted' => count($waitlisted),
        '%name' => $this->eventInstance->title->value,
        '@type' => $this->creationService->getRegistrationType() === RegistrationType::Series ? $this->t('series') : $this->t('event'),
      ]),
    ];

    $form['type'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Who would you like to contact?'),
      '#options' => [
        'registrants' => $this->t('Registrants'),
        'waitlist' => $this->t('Waitlisted Users'),
      ],
      '#default_value' => ['registrants'],
      '#required' => TRUE,
    ];

    $form['subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Email Subject'),
      '#description' => $this->t('Enter the subject of the email to send to the registrants.'),
      '#required' => TRUE,
    ];

    $form['message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Email Message'),
      '#description' => $this->t('Enter the message of the email to send to the registrants.'),
      '#required' => TRUE,
    ];

    $form['tokens'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['form-item'],
      ],
      'tokens' => $this->notificationService->getAvailableTokens(),
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Send Email(s)'),
    ];

    $link = Link::fromTextAndUrl($this->t('Go Back to Registration List'), new Url('entity.registrant.instance_listing', [
      'eventinstance' => $this->eventInstance->id(),
    ]));

    $form['back_link'] = [
      '#type' => 'markup',
      '#prefix' => '<span class="register-back-link">',
      '#markup' => $link->toString(),
      '#suffix' => '</span>',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();

    $params = [
      'subject' => $values['subject'],
      'body' => $values['message'],
    ];

    $registered = $values['type']['registrants'] === 'registrants' ? TRUE : FALSE;
    $waitlisted = $values['type']['waitlist'] === 'waitlist' ? TRUE : FALSE;

    $registrants = $this->creationService->retrieveRegisteredParties($registered, $waitlisted);

    $reg_count = $wait_count = 0;

    if (!empty($registrants)) {
      foreach ($registrants as $registrant) {
        $params['registrant'] = $registrant;

        $to = $registrant->email->value;
        $this->mail->mail('recurring_events_registration', 'custom', $to, $this->languageManager->getDefaultLanguage()->getId(), $params);

        if ($registrant->getWaitlist()) {
          $wait_count++;
        }
        else {
          $reg_count++;
        }
      }

      $this->messenger()->addMessage($this->t('Successfully sent emails to %reg_count registrants and %wait_count waitlisted users.', [
        '%reg_count' => $reg_count,
        '%wait_count' => $wait_count,
      ]));
    }
    else {
      $this->messenger()->addMessage($this->t('No emails were sent as there were no registrants or waitlist users to contact.'));
    }
  }

}
