<?php

declare(strict_types=1);

namespace Drupal\recurring_events_registration\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\ContentEntityDeleteForm;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;
use Drupal\recurring_events_registration\RegistrationCreationService;

/**
 * Provides a form for deleting Registrant entities.
 *
 * @ingroup recurring_events_registration
 */
class RegistrantDeleteForm extends ContentEntityDeleteForm {

  use AutowireTrait;

  public function __construct(
    EntityRepositoryInterface $entity_repository,
    EntityTypeBundleInfoInterface $entity_type_bundle_info,
    TimeInterface $time,
    protected RendererInterface $renderer,
    protected RegistrationCreationService $creationService,
  ) {
    parent::__construct($entity_repository, $entity_type_bundle_info, $time);
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Cancel Your Registration');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    /** @var \Drupal\recurring_events_registration\Entity\Registrant $entity */
    $entity = $this->entity;

    $build['cancel'] = [
      '#type' => 'container',
      '#weight' => -99,
      'title' => [
        '#type' => 'markup',
        '#prefix' => '<h2 class="registration-register-title">',
        '#markup' => $this->t('Cancel Event Registration'),
        '#suffix' => '</h2>',
      ],
      'intro' => [
        '#type' => 'markup',
        '#prefix' => '<p class=registration-register-intro">',
        '#markup' => $this->t('You are cancelling your registration for %email for %event. Once you do this, there may no longer be any spaces left for this event and you may not be able to register again.', [
          '%email' => $entity->email->value,
          '%event' => $entity->getEventSeries()?->title->value ?? '',
        ]),
        '#suffix' => '</p>',
      ],
    ];

    return $this->renderer->render($build);
  }

  /**
   * {@inheritdoc}
   *
   * If the delete command is canceled, return to the eventinstance list.
   */
  public function getCancelUrl() {
    return new Url('entity.eventinstance.canonical', ['eventinstance' => $this->getEntity()->getEventInstance()->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelText() {
    return $this->t('Go Back - Keep Registration');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Confirm Cancellation');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\recurring_events_registration\Entity\Registrant $entity */
    $entity = $this->entity;
    $entity->delete();
    $eventinstance = $entity->getEventInstance();

    $form_state->setRedirectUrl($eventinstance->toUrl('canonical'));

    $this->creationService->setEventInstance($eventinstance);
    if ($this->creationService->hasWaitlist() && $entity->waitlist->value == '0') {
      $this->creationService->promoteFromWaitlist();
    }

    $this->messenger()->addMessage($this->getDeletionMessage());
    $this->logDeletionMessage();
  }

  /**
   * {@inheritdoc}
   */
  protected function getDeletionMessage() {
    /** @var \Drupal\recurring_events\EventInterface $entity */
    $entity = $this->getEntity();

    return $this->t('Your registration for %email for %event has been cancelled.', [
      '%email' => $entity->email->value,
      '%event' => $entity->getEventSeries()->title->value,
    ]);
  }

}
