<?php

declare(strict_types=1);

namespace Drupal\recurring_events_registration\Form;

use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Render\Renderer;
use Drupal\Core\Url;
use Drupal\recurring_events\Entity\EventInstance;
use Drupal\recurring_events_registration\Entity\Registrant;
use Drupal\recurring_events_registration\NotificationService;
use Drupal\recurring_events_registration\RegistrationCreationService;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Provides a form for resending Registrant registration emails.
 *
 * @ingroup recurring_events_registration
 */
class RegistrantResendForm extends FormBase {

  use AutowireTrait;

  /**
   * The event instance object.
   *
   * @var \Drupal\recurring_events\Entity\EventInstance|mixed
   */
  protected EventInstance $eventInstance;

  /**
   * The registrant object.
   *
   * @var \Drupal\recurring_events_registration\Entity\Registrant
   */
  protected Registrant $registrant;

  public function __construct(
    RequestStack $requestStack,
    protected RegistrationCreationService $creationService,
    protected NotificationService $notificationService,
    protected MailManagerInterface $mail,
    protected Renderer $renderer,
    protected LanguageManagerInterface $languageManager,
  ) {
    $this->requestStack = $requestStack;

    $request = $this->getRequest();
    $params = $request->attributes->all();
    if (!empty($params['eventinstance']) && !empty($params['registrant'])) {
      $this->eventInstance = $params['eventinstance'];
      $this->creationService->setEventInstance($this->eventInstance);

      $this->registrant = $params['registrant'];
    }
    else {
      throw new NotFoundHttpException();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'recurring_events_registration_resend_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $key = $this->registrant->getWaitlist() ? 'waitlist_notification' : 'registration_notification';
    $this->notificationService->setEntity($this->registrant)->setKey($key);

    $form['resend'] = [
      '#type' => 'container',
      '#weight' => -99,
      'title' => [
        '#type' => 'markup',
        '#prefix' => '<h2 class="registration-register-title">',
        '#markup' => $this->t('Resend Registration Email'),
        '#suffix' => '</h2>',
      ],
      'intro' => [
        '#type' => 'markup',
        '#prefix' => '<p class=registration-register-intro">',
        '#markup' => $this->t('You are resending the registration email for %email for %event.', [
          '%email' => $this->registrant->email->value,
          '%event' => $this->eventInstance->title->value,
        ]),
        '#suffix' => '</p>',
      ],
    ];

    $form['subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Email Subject'),
      '#description' => $this->t('The subject of the email to send to the registrant'),
      '#default_value' => $this->notificationService->getSubject(FALSE),
      '#required' => TRUE,
    ];

    $form['message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Email Message'),
      '#description' => $this->t('The message for the email to send to the registrant'),
      '#default_value' => $this->notificationService->getMessage(FALSE),
      '#required' => TRUE,
    ];

    $form['tokens'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['form-item'],
      ],
      'tokens' => $this->notificationService->getAvailableTokens(),
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Resend Email'),
    ];

    $link = Link::fromTextAndUrl($this->t('Go Back to Registration List'), new Url('entity.registrant.instance_listing', [
      'eventinstance' => $this->eventInstance->id(),
    ]));

    $form['back_link'] = [
      '#type' => 'markup',
      '#prefix' => '<span class="register-back-link">',
      '#markup' => $link->toString(),
      '#suffix' => '</span>',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();

    $params = [
      'subject' => $values['subject'],
      'body' => $values['message'],
      'registrant' => $this->registrant,
    ];

    $to = $this->registrant->email->value;
    $this->mail->mail('recurring_events_registration', 'custom', $to, $this->languageManager->getDefaultLanguage()->getId(), $params);
    $this->messenger()->addMessage($this->t('Registrant email successfully resent.'));
  }

}
