<?php

declare(strict_types=1);

namespace Drupal\recurring_events_registration\Plugin\QueueWorker;

use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The 'recurring_events_registration_email_notifications_queue_worker'.
 */
#[QueueWorker(
  id: "recurring_events_registration_email_notifications_queue_worker",
  title: new TranslatableMarkup("Email Notifications Queue Worker"),
  cron: [
    "time" => 30,
  ],
)]
class EmailNotificationsQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  public function __construct(
    array $configuration,
    $plugin_id,
    array $plugin_definition,
    protected readonly MailManagerInterface $mailManager,
    protected readonly LanguageManagerInterface $languageManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.mail'),
      $container->get('language_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($item) {
    if (empty($item->key) || empty($item->to) || (empty($item->params['subject']) && empty($item->params['body']))) {
      return;
    }
    // All this worker has to do is to send the email.
    // The Subject and Body already have to be included in the `$item`.
    $this->mailManager->mail('recurring_events_registration', $item->key, $item->to, $this->languageManager->getDefaultLanguage()->getId(), $item->params);
  }

}
