<?php

declare(strict_types=1);

namespace Drupal\recurring_events_registration\Plugin\views\access;

use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\recurring_events_registration\AccessHandler;
use Drupal\views\Attribute\ViewsAccess;
use Drupal\views\Plugin\views\access\AccessPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Routing\Route;

/**
 * Custom access plugin for Event Registration views.
 *
 * @ingroup views_access_plugins
 *
 * @ViewsAccess(
 *   id = "event_registration_list_access",
 *   title = @Translation("View Event Registration List"),
 *   help = @Translation("Access will be granted depending on the registration configuration of an event.")
 * )
 */
#[ViewsAccess(
  id: "event_registration_list_access",
  title: new TranslatableMarkup("View Event Registration List"),
  help: new TranslatableMarkup("Access will be granted depending on the registration configuration of an event."),
)]
class EventRegistrationListAccess extends AccessPluginBase {

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly AccessHandler $accessHandler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('recurring_events_registration.access_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function summaryTitle() {
    return $this->t('Event Registration Access');
  }

  /**
   * {@inheritdoc}
   */
  public function access(AccountInterface $account) {
    return $this->accessHandler->eventHasRegistration() && $this->accessHandler->userHasPermission($account);
  }

  /**
   * {@inheritdoc}
   */
  public function alterRouteDefinition(Route $route) {
    $route->setRequirement('_custom_access', 'recurring_events_registration.access_handler::eventRegistrationListAccess');
  }

}
