<?php

declare(strict_types=1);

namespace Drupal\recurring_events_registration;

use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Entity\EntityFieldManager;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\recurring_events_registration\Enum\RegistrationType;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Defines a class to build a listing of Registrant entities.
 *
 * @ingroup recurring_events_registration
 */
class RegistrantListBuilder extends EntityListBuilder {

  public function __construct(
    EntityTypeInterface $entity_type,
    EntityStorageInterface $storage,
    protected readonly ConfigFactory $configFactory,
    protected readonly RequestStack $requestStack,
    protected readonly RegistrationCreationService $creationService,
    protected readonly EntityFieldManager $entityFieldManager,
  ) {
    parent::__construct($entity_type, $storage);

    $this->limit = $this->configFactory->get('recurring_events_registration.registrant.config')->get('limit');
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('config.factory'),
      $container->get('request_stack'),
      $container->get('recurring_events_registration.creation_service'),
      $container->get('entity_field.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['id'] = $this->t('Registrant ID');
    $header['series'] = $this->t('Series');
    $header['instance'] = $this->t('Instance');
    $header['type'] = $this->t('Type');
    $header['email'] = $this->t('Email');
    $header['waitlist'] = $this->t('Waitlist');
    $header['status'] = $this->t('Status');
    foreach ($this->getCustomFields() as $machine_name => $field) {
      $header[$machine_name] = $field;
    }
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var \Drupal\recurring_events_registration\Entity\Registrant $entity  */
    $series = $entity->getEventSeries();
    $instance = $entity->getEventInstance();

    $row['id'] = $entity->id();
    $row['series'] = $series?->toLink($series->title->value) ?? '';
    $timezone = new \DateTimeZone(date_default_timezone_get());
    $date = $instance?->date->start_date ?? '';
    $date->setTimezone($timezone);
    $row['instance'] = $instance?->toLink($date->format($this->configFactory->get('recurring_events_registration.registrant.config')->get('date_format'))) ?? '';
    $row['type'] = $entity->getRegistrationType()->getLabel();
    $row['email'] = $entity->get('email')->value;
    $row['waitlist'] = $entity->get('waitlist')->value ? $this->t('Yes') : $this->t('No');
    $row['status'] = $entity->get('status')->value ? $this->t('Complete') : $this->t('Pending');
    foreach ($this->getCustomFields($entity->bundle()) as $machine_name => $field) {
      $row[$machine_name] = $entity->get($machine_name)->value;
    }
    return $row + parent::buildRow($entity);
  }

  /**
   * Get custom fields.
   *
   * @param string $bundle
   *   The name of the entity bundle.
   *
   * @return array
   *   An array of custom fields.
   */
  protected function getCustomFields(string $bundle = 'default') {
    $custom_fields = [];
    $fields = $this->entityFieldManager->getFieldDefinitions('registrant', $bundle);
    foreach ($fields as $machine_name => $field) {
      if (strpos($machine_name, 'field_') === 0) {
        $custom_fields[$machine_name] = $field->label();
      }
    }
    return $custom_fields;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEntityIds() {
    $request = $this->requestStack->getCurrentRequest();
    $params = $request->attributes->all();

    $query = $this->getStorage()->getQuery()
      ->sort('changed', 'DESC');

    switch ($params['_route']) {
      case 'entity.registrant.instance_listing':
        $event_instance = $params['eventinstance'];
        $this->creationService->setEventInstance($event_instance);
        if ($this->creationService->getRegistrationType() === RegistrationType::Series) {
          $query->condition('eventseries_id', $event_instance->getEventSeries()->id());
        }
        else {
          $query->condition('eventinstance_id', $event_instance->id());
        }
        break;

      case 'registrations.user_tab':
        $user = $params['user'];
        $query->condition('user_id', $user->id());
        break;
    }

    // Only add the pager if a limit is specified.
    if ($this->limit) {
      $query->pager($this->limit);
    }
    return $query->accessCheck(FALSE)->execute();
  }

}
