<?php

declare(strict_types=1);

namespace Drupal\recurring_events\Controller;

use Drupal\Component\Utility\Xss;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;
use Drupal\recurring_events\EventInstanceStorageTrait;
use Drupal\recurring_events\EventInterface;
use Drupal\system\SystemManager;

/**
 * The EventInstanceController class.
 */
class EventInstanceController extends ControllerBase implements ContainerInjectionInterface {

  use AutowireTrait;
  use EventInstanceStorageTrait;

  /**
   * The current language code.
   *
   * @var string
   */
  protected string $langCode;

  public function __construct(
    protected readonly DateFormatterInterface $dateFormatter,
    protected readonly RendererInterface $renderer,
    protected readonly SystemManager $systemManager,
    EntityTypeManagerInterface $entityTypeManager,
  ) {
    $this->langCode = $this->languageManager()->getCurrentLanguage()->getId();
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * Get the page title for an eventinstance.
   *
   * @param \Drupal\recurring_events\EventInterface $eventinstance
   *   A eventinstance object.
   *
   * @return string
   *   The title of the page.
   */
  public function getTitle(EventInterface $eventinstance) {
    $title = '';
    try {
      $value = $eventinstance->get('title')->getValue();
      if (!empty($value[0]['value'])) {
        $title = $value[0]['value'];
      }

      if ($eventinstance->hasTranslation($this->langCode)) {
        $value = $eventinstance->getTranslation($this->langCode)->get('title')->getValue();
        if (!empty($value[0]['value'])) {
          $title = $value[0]['value'];
        }
      }
    }
    catch (\InvalidArgumentException $e) {
      $title = $eventinstance->getEventSeries()->label();
    }
    return $title;
  }

  /**
   * Displays an eventinstance revision.
   *
   * @param int $eventinstance_revision
   *   The eventinstance revision ID.
   *
   * @return array
   *   An array suitable for drupal_render().
   */
  public function revisionShow($eventinstance_revision) {
    $eventinstance = $this->eventInstanceStorage()->loadRevision($eventinstance_revision);
    $view_builder = $this->entityTypeManager()->getViewBuilder('eventinstance');

    return $view_builder->view($eventinstance);
  }

  /**
   * Page title callback for an eventinstance revision.
   *
   * @param int $eventinstance_revision
   *   The eventinstance revision ID.
   *
   * @return string
   *   The page title.
   */
  public function revisionPageTitle($eventinstance_revision) {
    $eventinstance = $this->eventInstanceStorage()->loadRevision($eventinstance_revision);
    return $this->t('Revision of %title from %date', [
      '%title' => $eventinstance->label(),
      '%date' => $this->dateFormatter->format($eventinstance->getRevisionCreationTime()),
    ]);
  }

  /**
   * Generates an overview table of older revisions of an eventinstance.
   *
   * @param \Drupal\recurring_events\EventInterface $eventinstance
   *   A eventinstance object.
   *
   * @return array
   *   An array as expected by drupal_render().
   */
  public function revisionOverview(EventInterface $eventinstance) {
    $account = $this->currentUser();
    $langcode = $eventinstance->language()->getId();
    $langname = $eventinstance->language()->getName();
    $languages = $eventinstance->getTranslationLanguages();
    $has_translations = (count($languages) > 1);

    $build['#title'] = $has_translations ? $this->t('@langname revisions for %title', [
      '@langname' => $langname,
      '%title' => $eventinstance->label(),
    ]) : $this->t('Revisions for %title', [
      '%title' => $eventinstance->label(),
    ]);
    $header = [$this->t('Revision'), $this->t('Operations')];

    $revert_permission = (($account->hasPermission("revert all eventinstance revisions") || $account->hasPermission('administer eventinstance entities')));
    $delete_permission = (($account->hasPermission("delete all eventinstance revisions") || $account->hasPermission('administer eventinstance entities')));

    $rows = [];

    $vids = $this->eventInstanceStorage()->revisionIds($eventinstance);

    $latest_revision = TRUE;

    foreach (array_reverse($vids) as $vid) {
      /** @var \Drupal\recurring_events\EventInterface $revision */
      $revision = $this->eventInstanceStorage()->loadRevision($vid);
      // Only show revisions that are affected by the language that is being
      // displayed.
      if ($revision->hasTranslation($langcode) && $revision->getTranslation($langcode)->isRevisionTranslationAffected()) {
        $username = [
          '#theme' => 'username',
          '#account' => $revision->getRevisionUser(),
        ];

        // Use revision link to link to revisions that are not active.
        $date = $this->dateFormatter->format($revision->getRevisionCreationTime(), 'short');
        if ($vid != $eventinstance->getRevisionId()) {
          $link = Link::fromTextAndUrl($date, new Url('entity.eventinstance.revision', [
            'eventinstance' => $eventinstance->id(),
            'eventinstance_revision' => $vid,
          ]));
        }
        else {
          $link = $eventinstance->toLink($date);
        }

        $row = [];
        $column = [
          'data' => [
            '#type' => 'inline_template',
            '#template' => '{% trans %}{{ date }} by {{ username }}{% endtrans %}{% if message %}<p class="revision-log">{{ message }}</p>{% endif %}',
            '#context' => [
              'date' => $link->toString(),
              'username' => $this->renderer->renderInIsolation($username),
              'message' => [
                '#markup' => $revision->getRevisionLogMessage(),
                '#allowed_tags' => Xss::getHtmlTagList(),
              ],
            ],
          ],
        ];
        // @todo Simplify once https://www.drupal.org/node/2334319 lands.
        $this->renderer->addCacheableDependency($column['data'], $username);
        $row[] = $column;

        if ($latest_revision) {
          $row[] = [
            'data' => [
              '#prefix' => '<em>',
              '#markup' => $this->t('Current revision'),
              '#suffix' => '</em>',
            ],
          ];
          foreach ($row as &$current) {
            $current['class'] = ['revision-current'];
          }
          $latest_revision = FALSE;
        }
        else {
          $links = [];
          if ($revert_permission) {
            $links['revert'] = [
              'title' => $this->t('Revert'),
              'url' => $has_translations ?
              Url::fromRoute('entity.eventinstance.revision_revert_translation_confirm', [
                'eventinstance' => $eventinstance->id(),
                'eventinstance_revision' => $vid,
                'langcode' => $langcode,
              ]) :
              Url::fromRoute('entity.eventinstance.revision_revert', [
                'eventinstance' => $eventinstance->id(),
                'eventinstance_revision' => $vid,
              ]),
            ];
          }

          if ($delete_permission) {
            $links['delete'] = [
              'title' => $this->t('Delete'),
              'url' => Url::fromRoute('entity.eventinstance.revision_delete', [
                'eventinstance' => $eventinstance->id(),
                'eventinstance_revision' => $vid,
              ]),
            ];
          }

          $row[] = [
            'data' => [
              '#type' => 'operations',
              '#links' => $links,
            ],
          ];
        }

        $rows[] = $row;
      }
    }

    $build['eventinstance_revisions_table'] = [
      '#theme' => 'table',
      '#rows' => $rows,
      '#header' => $header,
    ];

    return $build;
  }

}
