<?php

declare(strict_types=1);

namespace Drupal\recurring_events\EventSubscriber;

use Drupal\Core\Config\ConfigCrudEvent;
use Drupal\Core\Config\ConfigEvents;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Routing\RouteBuilderInterface;
use Drupal\Core\Routing\RouteSubscriberBase;
use Symfony\Component\Routing\RouteCollection;

/**
 * Sets the _admin_route for specific recurring events related routes.
 */
class RecurringEventsAdminRouteSubscriber extends RouteSubscriberBase {

  public function __construct(
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly RouteBuilderInterface $routerBuilder,
  ) {}

  /**
   * {@inheritdoc}
   */
  protected function alterRoutes(RouteCollection $collection) {
    if ($this->configFactory->get('node.settings')->get('use_admin_theme')) {
      foreach ($collection->all() as $route) {
        if ($route->hasOption('_recurring_events_operation_route')) {
          $route->setOption('_admin_route', TRUE);
        }
      }
    }
  }

  /**
   * Rebuilds the router when node.settings:use_admin_theme is changed.
   *
   * @param \Drupal\Core\Config\ConfigCrudEvent $event
   *   The config crud event that gets fired.
   */
  public function onConfigSave(ConfigCrudEvent $event) {
    if ($event->getConfig()->getName() === 'node.settings' && $event->isChanged('use_admin_theme')) {
      $this->routerBuilder->setRebuildNeeded();
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = parent::getSubscribedEvents();
    $events[ConfigEvents::SAVE][] = ['onConfigSave', 0];
    return $events;
  }

}
