<?php

declare(strict_types=1);

namespace Drupal\recurring_events\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Form controller for the eventinstance entity edit forms.
 *
 * @ingroup recurring_events
 */
class EventInstanceForm extends ContentEntityForm {

  use AutowireTrait;

  public function __construct(
    EntityRepositoryInterface $entity_repository,
    EntityTypeBundleInfoInterface $entity_type_bundle_info,
    TimeInterface $time,
    protected AccountProxyInterface $currentUser,
  ) {
    parent::__construct($entity_repository, $entity_type_bundle_info, $time);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);

    /** @var \Drupal\recurring_events\Entity\EventInstance $entity */
    $event = $this->entity;

    $form['notifications'] = [
      '#type' => 'container',
      '#weight' => -100,
      '#attributes' => [
        'class' => ['event-notifications'],
      ],
    ];

    $form['notifications']['edit_message'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['event-notification-message'],
      ],
      'title' => [
        '#type' => 'markup',
        '#prefix' => '<h3 class="event-notice-title">',
        '#markup' => $this->t('Data Inheritance'),
        '#suffix' => '</h3>',
      ],
      'message' => [
        '#type' => 'markup',
        '#prefix' => '<p class="event-message">',
        '#markup' => $this->t('Some of the data for event instances is inherited from the event series that the instance belongs to. @link.', [
          '@link' => $event->getEventSeries() ? $event->getEventSeries()->toLink($this->t('Edit the series'), 'edit-form')->toString() : '',
        ]),
        '#suffix' => '</p>',
      ],
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $entity = $this->getEntity();

    // Save as a new revision if requested to do so.
    if (!$form_state->isValueEmpty('revision') && $form_state->getValue('revision') != FALSE) {
      $entity->setNewRevision();

      // If a new revision is created, save the current user as revision author.
      $entity->setRevisionCreationTime($this->time->getRequestTime());
      $entity->setRevisionUserId($this->currentUser->id());
    }
    else {
      $entity->setNewRevision(FALSE);
    }

    $status = parent::save($form, $form_state);

    if ($entity->isDefaultTranslation()) {
      $message = $this->t('Event instance of %label has been saved.', [
        '%label' => $entity->getEventSeries()->title->value,
      ]);
    }
    else {
      $message = $this->t('@language translation of the Event Instance %label has been saved.', [
        '@language' => $entity->language()->getName(),
        '%label' => $entity->getUntranslated()->getEventSeries()->title->value,
      ]);
    }
    $this->messenger()->addMessage($message);

    $form_state->setRedirect('entity.eventinstance.canonical', ['eventinstance' => $entity->id()]);

    return $status;
  }

}
