<?php

declare(strict_types=1);

namespace Drupal\recurring_events\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\recurring_events\EventInstanceStorageTrait;

/**
 * Provides a form for deleting a eventinstance revision.
 *
 * @ingroup recurring_events
 */
class EventInstanceRevisionDeleteForm extends ConfirmFormBase {

  use AutowireTrait;
  use EventInstanceStorageTrait;
  use StringTranslationTrait;

  /**
   * The eventinstance revision.
   *
   * @var \Drupal\recurring_events\EventInterface
   */
  protected $revision;

  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    protected Connection $connection,
    protected DateFormatterInterface $dateFormatter,
  ) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'eventinstance_revision_delete_confirm';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete the revision from %revision-date?', [
      '%revision-date' => $this->dateFormatter->format($this->revision->getRevisionCreationTime()),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('entity.eventinstance.version_history', ['eventinstance' => $this->revision->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $eventinstance_revision = NULL) {
    $this->revision = $this->eventInstanceStorage()->loadRevision($eventinstance_revision);
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->eventInstanceStorage()->deleteRevision($this->revision->getRevisionId());

    $this->logger('content')->notice('eventinstance: deleted %title revision %revision.', [
      '%title' => $this->revision->label(),
      '%revision' => $this->revision->getRevisionId(),
    ]);
    $this->messenger()->addMessage($this->t('Revision from %revision-date of eventinstance %title has been deleted.', [
      '%revision-date' => $this->dateFormatter->format($this->revision->getRevisionCreationTime()),
      '%title' => $this->revision->label(),
    ]));
    $form_state->setRedirect(
      'entity.eventinstance.canonical',
       ['eventinstance' => $this->revision->id()]
    );
    if ($this->connection->query('SELECT COUNT(DISTINCT vid) FROM {eventinstance_field_revision} WHERE id = :id', [':id' => $this->revision->id()])->fetchField() > 1) {
      $form_state->setRedirect(
        'entity.eventinstance.version_history',
         ['eventinstance' => $this->revision->id()]
      );
    }
  }

}
