<?php

declare(strict_types=1);

namespace Drupal\recurring_events\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\recurring_events\EventInterface;
use Drupal\recurring_events\EventSeriesStorageTrait;

/**
 * Provides a form for deleting an eventseries revision.
 *
 * @ingroup recurring_events
 */
class EventSeriesRevisionDeleteForm extends ConfirmFormBase {

  use AutowireTrait;
  use EventSeriesStorageTrait;
  use StringTranslationTrait;

  /**
   * The eventseries revision.
   *
   * @var \Drupal\recurring_events\EventInterface
   */
  protected EventInterface $revision;

  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    protected Connection $connection,
    protected DateFormatterInterface $dateFormatter,
  ) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'eventseries_revision_delete_confirm';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete the revision from %revision-date?', [
      '%revision-date' => $this->dateFormatter->format($this->revision->getRevisionCreationTime()),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('entity.eventseries.version_history', ['eventseries' => $this->revision->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $eventseries_revision = NULL) {
    $this->revision = $this->eventSeriesStorage()->loadRevision($eventseries_revision);
    $form = parent::buildForm($form, $form_state);

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->eventSeriesStorage()->deleteRevision($this->revision->getRevisionId());

    $this->logger('content')->notice('eventseries: deleted %title revision %revision.', [
      '%title' => $this->revision->label(),
      '%revision' => $this->revision->getRevisionId(),
    ]);
    $this->messenger()->addMessage($this->t('Revision from %revision-date of eventseries %title has been deleted.', [
      '%revision-date' => $this->dateFormatter->format($this->revision->getRevisionCreationTime()),
      '%title' => $this->revision->label(),
    ]));
    $form_state->setRedirect(
      'entity.eventseries.canonical',
       ['eventseries' => $this->revision->id()]
    );
    if ($this->connection->query('SELECT COUNT(DISTINCT vid) FROM {eventseries_field_revision} WHERE id = :id', [':id' => $this->revision->id()])->fetchField() > 1) {
      $form_state->setRedirect(
        'entity.eventseries.version_history',
         ['eventseries' => $this->revision->id()]
      );
    }
  }

}
