<?php

declare(strict_types=1);

namespace Drupal\recurring_events\Form;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityConfirmFormBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Menu\MenuLinkManagerInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Builds the form to delete Event series type entities.
 */
class EventSeriesTypeDeleteForm extends EntityConfirmFormBase {

  use AutowireTrait;

  public function __construct(
    ModuleHandlerInterface $moduleHandler,
    EntityTypeManagerInterface $entityTypeManager,
    #[Autowire(service: 'cache.menu')]
    protected CacheBackendInterface $cacheMenu,
    protected MenuLinkManagerInterface $menuLinkManager,
  ) {
    $this->moduleHandler = $moduleHandler;
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete %name?', ['%name' => $this->entity->label()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    $descriptions = [];
    $instance_type = $this->entityTypeManager->getStorage('eventinstance_type')->load($this->entity->id());

    if (!empty($instance_type)) {
      $descriptions[] = $this->t('Deleting this event series type will also delete the corresponding event instance type.');
    }

    if ($this->moduleHandler->moduleExists('recurring_events_registration')) {
      $registrant_types = $this->entityTypeManager->getStorage('registrant_type')->load($this->entity->id());
      if (!empty($registrant_types)) {
        $descriptions[] = $this->t('Deleting this event series type will also delete the corresponding registrant type.');
      }
    }

    return implode("\r\n", $descriptions);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('entity.eventseries_type.collection');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->entity->delete();

    $this->messenger()->addMessage(
      $this->t('Successfully deleted @type: @label.',
        [
          '@type' => $this->entity->bundle(),
          '@label' => $this->entity->label(),
        ]
      )
    );

    $this->cacheMenu->deleteAll();
    $this->menuLinkManager->rebuild();

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
