<?php

declare(strict_types=1);

namespace Drupal\recurring_events\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Hook implementations for recurring_events.
 */
class RecurringEventsTokensHooks {

  use StringTranslationTrait;

  public function __construct(
    protected readonly ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * Implements hook_views_token_info().
   */
  #[Hook('token_info')]
  public function tokenInfo(): array {
    $eventinstance = [];
    $eventinstance_type = [
      'name' => $this->t('Event Instance'),
      'description' => $this->t('Tokens for the eventinstance entity type.'),
      'needs-data' => 'eventinstance',
    ];

    $eventinstance['title'] = [
      'name' => $this->t('Event Instance Inherited Title'),
      'description' => $this->t('The title of the event instance.'),
    ];

    $eventinstance['description'] = [
      'name' => $this->t('Event Instance Inherited Description'),
      'description' => $this->t('The description of the event instance.'),
    ];

    $eventinstance['date'] = [
      'name' => $this->t('Event Instance Start Date'),
      'description' => $this->t('The start date of the event instance.'),
    ];

    $eventinstance['end_date'] = [
      'name' => $this->t('Event Instance End Date'),
      'description' => $this->t('The end date of the event instance.'),
    ];

    $eventinstance['url'] = [
      'name' => $this->t('Event Instance URL'),
      'description' => $this->t('The URL of the event instance.'),
    ];

    $eventseries = [];
    $eventseries_type = [
      'name' => $this->t('Event Series'),
      'description' => $this->t('Tokens for the eventseries entity type.'),
      'needs-data' => 'eventseries',
    ];

    $eventseries['title'] = [
      'name' => $this->t('Event Series Title'),
      'description' => $this->t('The title of the event series.'),
    ];

    $eventseries['body'] = [
      'name' => $this->t('Event Series Body'),
      'description' => $this->t('The description of the event series.'),
    ];

    return [
      'types' => [
        'eventinstance' => $eventinstance_type,
        'eventseries' => $eventseries_type,
      ],
      'tokens' => [
        'eventinstance' => $eventinstance,
        'eventseries' => $eventseries,
      ],
    ];
  }

  /**
   * Implements hook_views_tokens().
   */
  #[Hook('tokens')]
  public function tokens($type, $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata): array {
    $replacements = [];
    if ($type == 'eventinstance' && !empty($data['eventinstance'])) {
      $event_instance = $data['eventinstance'];
      $timezone = new \DateTimeZone(date_default_timezone_get());
      foreach ($tokens as $name => $original) {
        $date_format = $this->configFactory->get('recurring_events.eventinstance.config')->get('date_format');

        // Extract the date format if provided.
        $custom_date_format_indicator = ":custom:";
        if (str_contains($name, $custom_date_format_indicator)) {
          $date_format = substr($name, strpos($name, $custom_date_format_indicator) + strlen($custom_date_format_indicator));
          $name = substr($name, 0, strpos($name, $custom_date_format_indicator));
        }

        switch ($name) {
          case 'title':
            $replacements[$original] = ($event_instance->hasField('title')) ? $event_instance->title->value : '';
            break;

          case 'description':
            $replacements[$original] = ($event_instance->hasField('description')) ? $event_instance->description->value : '';
            break;

          case 'date':
            $replacements[$original] = $event_instance->date->start_date->setTimezone($timezone)->format($date_format);
            break;

          case 'end_date':
            $replacements[$original] = $event_instance->date->end_date->setTimezone($timezone)->format($date_format);
            break;

          case 'url':
            $replacements[$original] = $event_instance->toUrl('canonical')->setAbsolute(TRUE)->toString();
            break;
        }
      }
    }
    if ($type == 'eventseries' && !empty($data['eventseries'])) {
      $event_series = $data['eventseries'];
      foreach ($tokens as $name => $original) {
        switch ($name) {
          case 'title':
            $replacements[$original] = $event_series->title->value;
            break;

          case 'body':
            $replacements[$original] = $event_series->body->value;
            break;

        }
      }
    }
    return $replacements;
  }

}
