<?php

declare(strict_types=1);

namespace Drupal\recurring_events\Plugin\views\field;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\views\Attribute\ViewsField;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Field handler to show the start date of an event series.
 *
 * @ingroup views_field_handlers
 */
#[ViewsField(
  id: "eventseries_start_date",
)]
class EventSeriesStartDate extends FieldPluginBase {

  public function __construct(
    array $configuration,
    string $plugin_id,
    $plugin_definition,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly Connection $database,
    protected readonly PluginManagerInterface $joinHandler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    /** @var \Drupal\Core\Config\ConfigFactoryInterface $config_factory */
    $config_factory = $container->get('config.factory');
    /** @var \Drupal\Core\Database\Connection $database */
    $database = $container->get('database');
    /** @var \Drupal\Component\Plugin\PluginManagerInterface $join_handler */
    $join_handler = $container->get('plugin.manager.views.join');
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $config_factory,
      $database,
      $join_handler
    );
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    // Add a subquery to get the earliest instance start date for a series.
    $subQuery = $this->database->select('eventinstance_field_data', 'eventinstance_field_data');
    $subQuery->addField('eventinstance_field_data', 'eventseries_id');
    $subQuery->addExpression("MIN(eventinstance_field_data.date__value)", 'eventseries_start_date');
    $subQuery->groupBy("eventinstance_field_data.eventseries_id");

    // Create a join for the subquery.
    $joinDefinition = [
      'table formula' => $subQuery,
      'field' => 'eventseries_id',
      'left_table' => 'eventseries_field_data',
      'left_field' => 'id',
      'adjust' => TRUE,
    ];

    // Add the subquery join to the main query.
    /** @var \Drupal\views\Plugin\views\join\JoinPluginBase $join */
    $join = $this->joinHandler->createInstance('standard', $joinDefinition);
    $this->query->addRelationship('eventseries_start_date', $join, 'eventseries_field_data');

    // Add the field to the view.
    $this->query->addField(NULL, 'eventseries_start_date', 'eventseries_start_date');
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    if (!isset($values->eventseries_start_date)) {
      return 'N/A';
    }

    $date = new DrupalDateTime($values->eventseries_start_date, 'UTC');
    $format = $this->configFactory->get('recurring_events.eventseries.config')->get('date_format');
    return $date->format($format, [
      'timezone' => date_default_timezone_get(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function clickSort($order) {
    if (isset($this->field_alias)) {
      $params = $this->options['group_type'] != 'group' ? ['function' => $this->options['group_type']] : [];
      $this->query->addOrderBy(NULL, 'eventseries_start_date', $order, $this->field_alias, $params);
    }
  }

}
