<?php

namespace Drupal\redirect_audit\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\redirect_audit\RedirectChainResolverInterface;

/**
 * Service for fixing redirect chains and removing loops.
 */
class RedirectAuditFixer {

  /**
   * The redirect chain resolver service.
   *
   * @var \Drupal\redirect_audit\RedirectChainResolverInterface
   */
  protected RedirectChainResolverInterface $chainResolver;

  /**
   * The redirect audit storage service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditStorage
   */
  protected RedirectAuditStorage $storage;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs a RedirectAuditFixer object.
   *
   * @param \Drupal\redirect_audit\RedirectChainResolverInterface $chain_resolver
   *   The redirect chain resolver service.
   * @param \Drupal\redirect_audit\Service\RedirectAuditStorage $storage
   *   The redirect audit storage service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   */
  public function __construct(
    RedirectChainResolverInterface $chain_resolver,
    RedirectAuditStorage $storage,
    EntityTypeManagerInterface $entity_type_manager,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $this->chainResolver = $chain_resolver;
    $this->storage = $storage;
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger_factory->get('redirect_audit');
  }

  /**
   * Fixes a specific redirect chain.
   *
   * @param int $audit_id
   *   The audit record ID.
   *
   * @return bool
   *   TRUE if the chain was fixed, FALSE otherwise.
   */
  public function fixChain(int $audit_id): bool {
    try {
      $record = $this->storage->getChainById($audit_id);
      if (!$record) {
        $this->logger->warning('Audit record @id not found', ['@id' => $audit_id]);
        return FALSE;
      }

      // Load the source redirect.
      $redirect = $this->entityTypeManager->getStorage('redirect')->load($record['source_rid']);
      if (!$redirect) {
        $this->logger->warning('Source redirect @rid not found, cleaning audit record', [
          '@rid' => $record['source_rid'],
        ]);
        $this->storage->deleteChain($audit_id);
        return FALSE;
      }

      // Fix the chain.
      $fixed = $this->chainResolver->checkAndResolveChain($redirect);
      if ($fixed) {
        $this->storage->deleteChain($audit_id);
        $this->logger->info('Fixed chain: @source → @target', [
          '@source' => $record['source_rid'],
          '@target' => $record['target_rid'],
        ]);
        return TRUE;
      }

      $this->logger->warning('Could not fix chain @source → @target', [
        '@source' => $record['source_rid'],
        '@target' => $record['target_rid'],
      ]);
      return FALSE;
    }
    catch (\Exception $e) {
      $this->logger->error('Error fixing chain @id: @message', [
        '@id' => $audit_id,
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Fixes all detected redirect chains.
   *
   * @return array
   *   Stats: fixed, errors, skipped, total.
   */
  public function fixAll(): array {
    $fixed = 0;
    $errors = 0;
    $skipped = 0;

    $chains = $this->storage->getChains();
    $total = count($chains);

    $this->logger->info('Starting bulk fix. Total chains: @total', [
      '@total' => $total,
    ]);

    foreach ($chains as $record) {
      // Skip loops.
      if ($record['source_rid'] === $record['target_rid']) {
        $skipped++;
        continue;
      }

      // Fix the chain.
      $result = $this->fixChain($record['id']);
      if ($result) {
        $fixed++;
      }
      else {
        $errors++;
      }

      // Log progress every 10 records.
      $processed = $fixed + $errors + $skipped;
      if ($processed % 10 === 0) {
        $this->logger->info('Progress: @processed/@total processed', [
          '@processed' => $processed,
          '@total' => $total,
        ]);
      }
    }

    $this->logger->info('Bulk fix complete. Fixed: @fixed, Errors: @errors, Skipped: @skipped', [
      '@fixed' => $fixed,
      '@errors' => $errors,
      '@skipped' => $skipped,
    ]);

    return [
      'fixed' => $fixed,
      'errors' => $errors,
      'skipped' => $skipped,
      'total' => $total,
    ];
  }

}
