<?php

namespace Drupal\redirect_audit\Plugin\QueueWorker;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\redirect_audit\Service\RedirectAuditAnalyzer;
use Drupal\redirect_audit\Service\RedirectAuditStorage;
use Drupal\redirect_audit\Service\RedirectAuditFixer;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes redirect audit queue items.
 *
 * @QueueWorker(
 *   id = "redirect_audit_queue",
 *   title = @Translation("Redirect Audit Queue Worker"),
 *   cron = {"time" = 60}
 * )
 */
class RedirectAuditQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * The redirect audit analyzer service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditAnalyzer
   */
  protected RedirectAuditAnalyzer $analyzer;

  /**
   * The redirect audit storage service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditStorage
   */
  protected RedirectAuditStorage $storage;

  /**
   * The redirect audit fixer service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditFixer
   */
  protected RedirectAuditFixer $fixer;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a RedirectAuditQueueWorker object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\redirect_audit\Service\RedirectAuditAnalyzer $analyzer
   *   The redirect audit analyzer service.
   * @param \Drupal\redirect_audit\Service\RedirectAuditStorage $storage
   *   The redirect audit storage service.
   * @param \Drupal\redirect_audit\Service\RedirectAuditFixer $fixer
   *   The redirect audit fixer service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    RedirectAuditAnalyzer $analyzer,
    RedirectAuditStorage $storage,
    RedirectAuditFixer $fixer,
    EntityTypeManagerInterface $entity_type_manager,
    LoggerInterface $logger,
    ConfigFactoryInterface $config_factory,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->analyzer = $analyzer;
    $this->storage = $storage;
    $this->fixer = $fixer;
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('redirect_audit.analyzer'),
      $container->get('redirect_audit.storage'),
      $container->get('redirect_audit.fixer'),
      $container->get('entity_type.manager'),
      $container->get('logger.factory')->get('redirect_audit'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    // Validate queue item data.
    if (!isset($data['rid']) || !isset($data['operation'])) {
      $this->logger->error('Invalid queue item: missing rid or operation');
      return;
    }

    $rid = $data['rid'];

    // Load the redirect entity.
    $redirect_storage = $this->entityTypeManager->getStorage('redirect');
    $redirect = $redirect_storage->load($rid);

    // If redirect was deleted, skip processing.
    if (!$redirect) {
      $this->logger->debug('Redirect @rid no longer exists, skipping queue item', ['@rid' => $rid]);
      return;
    }

    $config = $this->configFactory->get('redirect_audit.settings');

    // Analyze the redirect itself.
    $chain_data = $this->analyzer->analyzeRedirect($redirect);
    if ($chain_data) {
      $this->processChainData($chain_data, $rid, $config->get('autofix_enabled'));
    }

    // Find and analyze redirects that point TO this redirect.
    $chains = $this->analyzer->findAndAnalyzeRedirectsToSource($redirect);
    foreach ($chains as $chain_data) {
      $this->processChainData($chain_data, $rid, $config->get('autofix_enabled'), TRUE);
    }
  }

  /**
   * Processes and saves chain data, with optional auto-fix.
   *
   * @param array $chain_data
   *   The chain data array with source_rid, target_rid, and path.
   * @param int $rid
   *   The redirect ID being processed (for logging).
   * @param bool $autofix_enabled
   *   Whether to automatically fix the chain.
   * @param bool $is_pointing_chain
   *   Whether this is a chain pointing to the redirect (for logging).
   */
  private function processChainData(array $chain_data, int $rid, bool $autofix_enabled, bool $is_pointing_chain = FALSE): void {
    $audit_id = $this->storage->saveChain(
      $chain_data['source_rid'],
      $chain_data['target_rid'],
      $chain_data['path']
    );

    $log_message = $is_pointing_chain
      ? 'Chain detected (queue, pointing to @rid)'
      : 'Chain detected (queue) for redirect @rid';
    $this->logger->info($log_message, ['@rid' => $rid]);

    if ($autofix_enabled) {
      $this->fixer->fixChain($audit_id);
      $this->logger->info('Chain auto-fixed (queue): @id', ['@id' => $audit_id]);
    }
  }

}
