<?php

namespace Drupal\redirect_audit\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\redirect\Entity\Redirect;
use Drupal\redirect_audit\RedirectChainResolverInterface;

/**
 * Service for analyzing redirect chains and loops.
 */
class RedirectAuditAnalyzer {

  /**
   * The redirect chain resolver service.
   *
   * @var \Drupal\redirect_audit\RedirectChainResolverInterface
   */
  protected RedirectChainResolverInterface $chainResolver;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The redirect audit storage service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditStorage
   */
  protected RedirectAuditStorage $storage;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The maximum chain depth (cached).
   *
   * @var int|null
   */
  protected ?int $maxChainDepth = NULL;

  /**
   * Constructs a RedirectAuditAnalyzer object.
   *
   * @param \Drupal\redirect_audit\RedirectChainResolverInterface $chain_resolver
   *   The redirect chain resolver service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\redirect_audit\Service\RedirectAuditStorage $storage
   *   The redirect audit storage service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    RedirectChainResolverInterface $chain_resolver,
    EntityTypeManagerInterface $entity_type_manager,
    RedirectAuditStorage $storage,
    LoggerChannelFactoryInterface $logger_factory,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->chainResolver = $chain_resolver;
    $this->entityTypeManager = $entity_type_manager;
    $this->storage = $storage;
    $this->logger = $logger_factory->get('redirect_audit');
    $this->configFactory = $config_factory;
  }

  /**
   * Analyzes a redirect to detect if it forms part of a chain or loop.
   *
   * @param \Drupal\redirect\Entity\Redirect $redirect
   *   The redirect entity to analyze.
   *
   * @return array|null
   *   Chain information or NULL if none.
   */
  public function analyzeRedirect(Redirect $redirect): ?array {
    try {
      $chain = [$redirect];
      $visited_ids = [$redirect->id() => TRUE];
      $loop_detected = FALSE;
      $max_depth = $this->getMaxChainDepth();

      // Follow the chain of redirects.
      $current_redirect = $redirect;
      for ($depth = 0; $depth < $max_depth; $depth++) {
        $target_url = $current_redirect->getRedirectUrl();

        if (!$target_url) {
          break;
        }

        $path = '';
        $query = [];

        // Extract path and query from the destination URL.
        if ($target_url->isRouted()) {
          $path = $target_url->getInternalPath();
          $query = $target_url->getOption('query') ?: [];
        }
        else {
          $uri_string = $target_url->toString();
          // Skip external URLs.
          if (str_starts_with($uri_string, 'http://') || str_starts_with($uri_string, 'https://')) {
            break;
          }
          $path = ltrim($uri_string, '/');
        }

        if (empty($path)) {
          break;
        }

        $next_redirect = $this->findRedirectByPath(
          $path,
          $query,
          $current_redirect->language->value
        );

        if (!$next_redirect || $next_redirect->id() == $current_redirect->id()) {
          break;
        }

        // Loop detected.
        if (isset($visited_ids[$next_redirect->id()])) {
          $loop_detected = TRUE;
          $chain[] = $next_redirect;
          break;
        }
        $visited_ids[$next_redirect->id()] = TRUE;
        $chain[] = $next_redirect;
        $current_redirect = $next_redirect;
      }

      // No chain if only single redirect.
      if (count($chain) === 1) {
        return NULL;
      }

      // Verify that the source is not part of the loop.
      $is_real_loop = FALSE;
      if ($loop_detected) {
        $loop_point = $chain[count($chain) - 1]->id();
        $is_real_loop = ($chain[0]->id() === $loop_point);
      }

      return [
        'source_rid' => (int) $chain[0]->id(),
        'target_rid' => (int) $chain[count($chain) - 1]->id(),
        'path' => $this->calculatePath($chain),
        'is_loop' => $is_real_loop,
        'chain' => $chain,
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Error analyzing redirect @id: @message', [
        '@id' => $redirect->id(),
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Finds and analyzes redirects that point to the given redirect.
   *
   * @param \Drupal\redirect\Entity\Redirect $redirect
   *   The redirect to check.
   *
   * @return array
   *   Array of chain data arrays for any chains detected.
   */
  public function findAndAnalyzeRedirectsToSource(Redirect $redirect): array {
    $results = [];
    $storage = $this->entityTypeManager->getStorage('redirect');

    $source_path = $redirect->getSourcePathWithQuery();

    // Find all redirects that point to this source path.
    $pointing_redirects = $storage->loadByProperties([
      'redirect_redirect__uri' => 'internal:' . $source_path,
    ]);

    foreach ($pointing_redirects as $pointing_redirect) {
      $chain_data = $this->analyzeRedirect($pointing_redirect);
      if ($chain_data) {
        $results[] = $chain_data;
      }
    }

    return $results;
  }

  /**
   * Calculates the path string from a chain of redirects.
   *
   * @param array $chain
   *   Array of Redirect entities.
   *
   * @return string
   *   Dot-separated string of intermediate redirect IDs.
   */
  public function calculatePath(array $chain): string {
    // Exclude only the first redirect (source), but include all others.
    $intermediates = array_slice($chain, 1);
    $ids = array_map(fn($redirect) => $redirect->id(), $intermediates);
    return implode('.', $ids);
  }

  /**
   * Detects all redirect chains and loops.
   *
   * @return array
   *   Stats: redirects_analyzed, chains_found, loops_found.
   */
  public function detectChains(): array {
    $redirects_analyzed = 0;
    $chains_found = 0;
    $loops_found = 0;

    // Clear existing audit data to ensure we only show current state.
    $this->storage->clearProcessed();

    $storage = $this->entityTypeManager->getStorage('redirect');
    $redirects = $storage->loadMultiple();

    $this->logger->info('Starting chain detection. Total redirects: @count', [
      '@count' => count($redirects),
    ]);

    $loop_members = [];
    $processed_loops = [];
    $all_chain_data = [];

    // Analyze all redirects and identify loops.
    foreach ($redirects as $redirect) {
      try {
        $redirects_analyzed++;

        // Analyze each redirect to detect if it's part of a chain or loop.
        $chain_data = $this->analyzeRedirect($redirect);
        if ($chain_data) {
          $all_chain_data[] = $chain_data;

          if ($chain_data['is_loop']) {
            // Excluding duplicates.
            $loop_ids = array_map(fn($r) => $r->id(), $chain_data['chain']);
            array_pop($loop_ids);
            // Get unique IDs and sort to create a consistent signature.
            $unique_loop_ids = array_unique($loop_ids);
            sort($unique_loop_ids);
            $loop_signature = implode('-', $unique_loop_ids);

            // Skip if we've already processed this loop.
            if (isset($processed_loops[$loop_signature])) {
              continue;
            }

            // Mark all members of this loop.
            foreach ($unique_loop_ids as $loop_id) {
              $loop_members[$loop_id] = TRUE;
            }

            $processed_loops[$loop_signature] = TRUE;
            $chains_found++;
            $loops_found++;

            $this->storage->saveChain(
              $chain_data['source_rid'],
              $chain_data['target_rid'],
              $chain_data['path']
            );
          }
        }

        if ($redirects_analyzed % 100 === 0) {
          $this->logger->info('Progress: @count redirects analyzed', [
            '@count' => $redirects_analyzed,
          ]);
        }
      }
      catch (\Exception $e) {
        $this->logger->error('Error processing redirect @id: @message', [
          '@id' => $redirect->id(),
          '@message' => $e->getMessage(),
        ]);
      }
    }

    // List of intermediate redirects.
    $intermediate_redirects = [];
    foreach ($all_chain_data as $chain_data) {
      if (!$chain_data['is_loop'] && !empty($chain_data['path'])) {
        $path_ids = explode('.', $chain_data['path']);
        foreach ($path_ids as $path_id) {
          $intermediate_redirects[(int) $path_id] = TRUE;
        }
      }
    }

    // Save regular chains that don't touch loops.
    foreach ($all_chain_data as $chain_data) {
      try {
        if (!$chain_data['is_loop']) {
          if (isset($intermediate_redirects[$chain_data['source_rid']])) {
            continue;
          }

          $chain_touches_loop = FALSE;

          // Check if the target or intermediate redirects are loop members.
          if (isset($loop_members[$chain_data['target_rid']])) {
            $chain_touches_loop = TRUE;
          }

          if (!$chain_touches_loop && !empty($chain_data['path'])) {
            $path_ids = explode('.', $chain_data['path']);
            foreach ($path_ids as $path_id) {
              if (isset($loop_members[(int) $path_id])) {
                $chain_touches_loop = TRUE;
                break;
              }
            }
          }

          if (!$chain_touches_loop) {
            $chains_found++;
            $this->storage->saveChain(
              $chain_data['source_rid'],
              $chain_data['target_rid'],
              $chain_data['path']
            );
          }
        }
      }
      catch (\Exception $e) {
        $this->logger->error('Error processing chain: @message', [
          '@message' => $e->getMessage(),
        ]);
      }
    }

    $this->logger->info('Chain detection complete. Analyzed: @count, Chains: @chains, Loops: @loops', [
      '@count' => $redirects_analyzed,
      '@chains' => $chains_found,
      '@loops' => $loops_found,
    ]);

    return [
      'analyzed' => $redirects_analyzed,
      'chains_found' => $chains_found,
      'loops_found' => $loops_found,
    ];
  }

  /**
   * Gets the maximum chain depth from configuration.
   *
   * @return int
   *   The maximum chain depth.
   */
  protected function getMaxChainDepth(): int {
    if ($this->maxChainDepth === NULL) {
      $config = $this->configFactory->get('redirect.settings');
      $this->maxChainDepth = (int) $config->get('max_chain_depth') ?: 10;
    }
    return $this->maxChainDepth;
  }

  /**
   * Finds a redirect by path and query.
   *
   * @param string $path
   *   The path to search for.
   * @param array $query
   *   Query parameters.
   * @param string $language
   *   Language code.
   *
   * @return \Drupal\redirect\Entity\Redirect|null
   *   The redirect entity or NULL.
   */
  protected function findRedirectByPath(string $path, array $query, string $language): ?Redirect {
    $storage = $this->entityTypeManager->getStorage('redirect');
    $hash = Redirect::generateHash($path, $query, $language);
    $ids = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('hash', $hash)
      ->range(0, 1)
      ->execute();

    if (!$ids) {
      return NULL;
    }

    /** @var \Drupal\redirect\Entity\Redirect|null $redirect */
    $redirect = $storage->load(reset($ids));
    return $redirect;
  }

}
