<?php

namespace Drupal\redirect_audit\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Drupal\redirect_audit\RedirectAuditBatch;
use Drupal\redirect_audit\Service\RedirectAuditAnalyzer;
use Drupal\redirect_audit\Service\RedirectAuditFixer;
use Drupal\redirect_audit\Service\RedirectAuditStorage;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Redirect Audit Dashboard form.
 */
class RedirectAuditDashboardForm extends FormBase {

  /**
   * The redirect audit storage service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditStorage
   */
  protected RedirectAuditStorage $storage;

  /**
   * The redirect audit analyzer service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditAnalyzer
   */
  protected RedirectAuditAnalyzer $analyzer;

  /**
   * The redirect audit fixer service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditFixer
   */
  protected RedirectAuditFixer $fixer;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a RedirectAuditDashboardForm object.
   *
   * @param \Drupal\redirect_audit\Service\RedirectAuditStorage $storage
   *   The redirect audit storage service.
   * @param \Drupal\redirect_audit\Service\RedirectAuditAnalyzer $analyzer
   *   The redirect audit analyzer service.
   * @param \Drupal\redirect_audit\Service\RedirectAuditFixer $fixer
   *   The redirect audit fixer service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    RedirectAuditStorage $storage,
    RedirectAuditAnalyzer $analyzer,
    RedirectAuditFixer $fixer,
    ConfigFactoryInterface $config_factory,
    MessengerInterface $messenger,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    $this->storage = $storage;
    $this->analyzer = $analyzer;
    $this->fixer = $fixer;
    $this->setConfigFactory($config_factory);
    $this->setMessenger($messenger);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('redirect_audit.storage'),
      $container->get('redirect_audit.analyzer'),
      $container->get('redirect_audit.fixer'),
      $container->get('config.factory'),
      $container->get('messenger'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'redirect_audit_dashboard_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $stats = $this->storage->getStats();
    $config = $this->configFactory->get('redirect_audit.settings');

    $form['status'] = [
      '#type' => 'item',
      '#title' => $this->t('Status'),
      '#markup' => $this->buildStatusMessage($stats),
    ];

    // Actions.
    $form['actions'] = [
      '#type' => 'actions',
      'audit' => [
        '#type' => 'submit',
        '#value' => $this->t('Audit'),
        '#submit' => ['::auditSubmit'],
        '#button_type' => 'primary',
      ],
      'fix' => [
        '#type' => 'submit',
        '#value' => $this->t('Fix'),
        '#submit' => ['::fixSubmit'],
        '#disabled' => $stats['total'] === 0,
      ],
      'clear' => [
        '#type' => 'submit',
        '#value' => $this->t('Clear'),
        '#submit' => ['::clearSubmit'],
        '#disabled' => $stats['total'] === 0,
      ],
    ];

    $form['autofix_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Autofix enabled'),
      '#default_value' => $config->get('autofix_enabled'),
      '#ajax' => [
        'callback' => '::ajaxAutofixToggle',
      ],
    ];

    // Results table.
    if ($stats['total'] > 0) {
      $form['results'] = $this->buildResultsTable();
    }

    return $form;
  }

  /**
   * Builds the status message based on statistics.
   *
   * @param array $stats
   *   Statistics array with 'chains' and 'loops' keys.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The status message.
   */
  protected function buildStatusMessage(array $stats) {
    if ($stats['total'] === 0) {
      return $this->t('No scan launched yet. Click "Audit" to scan all redirects.');
    }

    $parts = [];
    if ($stats['chains'] > 0) {
      $parts[] = $this->formatPlural($stats['chains'], '1 chain', '@count chains');
    }
    if ($stats['loops'] > 0) {
      $parts[] = $this->formatPlural($stats['loops'], '1 loop', '@count loops');
    }

    return $this->t('Found @items. Click "Fix" to correct them or "Clear" to remove records.', [
      '@items' => implode(' and ', $parts),
    ]);
  }

  /**
   * Builds the results table with detected chains and loops.
   *
   * @return array
   *   A render array for the table element.
   */
  protected function buildResultsTable(): array {
    $chains = $this->storage->getChains();
    if (empty($chains)) {
      return [];
    }

    $redirect_ids = [];
    foreach ($chains as $chain) {
      $redirect_ids[$chain['source_rid']] = TRUE;
      $redirect_ids[$chain['target_rid']] = TRUE;
    }

    $redirect_storage = $this->entityTypeManager->getStorage('redirect');
    $redirects = $redirect_storage->loadMultiple(array_keys($redirect_ids));

    $rows = [];
    foreach ($chains as $chain) {
      $rows[] = $this->buildTableRow($chain, $redirects);
    }

    return [
      '#type' => 'table',
      '#caption' => $this->t('Audit Results'),
      '#header' => [
        $this->t('ID'),
        $this->t('Source Path'),
        $this->t('Intermediate Links'),
        $this->t('Target Path'),
        $this->t('Type'),
      ],
      '#rows' => $rows,
    ];
  }

  /**
   * Builds a single table row from chain data.
   *
   * @param array $chain
   *   Chain record with keys: id, source_rid, target_rid, path.
   * @param array $redirects
   *   Pre-loaded array of Redirect entities keyed by ID.
   *
   * @return array
   *   Array of table cell values.
   */
  protected function buildTableRow(array $chain, array $redirects): array {
    $source_redirect = $redirects[$chain['source_rid']] ?? NULL;
    $target_redirect = $redirects[$chain['target_rid']] ?? NULL;

    $source_path = $source_redirect?->getSourcePathWithQuery() ?? $this->t('N/A');

    $target_path = $this->t('N/A');
    if ($target_redirect && ($target_url = $target_redirect->getRedirectUrl())) {
      $target_path = $target_url->toString();
    }

    $intermediate = $this->formatChainPath($chain['path']);

    $type = $chain['source_rid'] === $chain['target_rid'] ? $this->t('Loop') : $this->t('Chain');

    return [
      $chain['id'],
      $source_path,
      ['data' => $intermediate],
      $target_path,
      $type,
    ];
  }

  /**
   * Formats intermediate redirect IDs with links.
   *
   * @param string $path
   *   Chain paths (dot-separated redirect IDs).
   *
   * @return array
   *   Render array with redirect ID links separated by arrows.
   */
  protected function formatChainPath(string $path): array {
    $ids = array_filter(array_map('intval', explode('.', $path)));

    if (empty($ids)) {
      return ['#markup' => '-'];
    }

    $elements = [];
    foreach ($ids as $index => $id) {
      if ($index > 0) {
        $elements[] = ['#markup' => ' → '];
      }
      $elements[] = [
        '#type' => 'link',
        '#title' => $id,
        '#url' => Url::fromRoute('entity.redirect.edit_form', ['redirect' => $id]),
      ];
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {}

  /**
   * Submit handler for audit button.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function auditSubmit(array &$form, FormStateInterface $form_state): void {
    $batch = [
      'title' => $this->t('Auditing Redirects'),
      'operations' => [
        [[RedirectAuditBatch::class, 'processAudit'], []],
      ],
      'finished' => [RedirectAuditBatch::class, 'finishedAudit'],
    ];

    batch_set($batch);
  }

  /**
   * Submit handler for fix button.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function fixSubmit(array &$form, FormStateInterface $form_state): void {
    $batch = [
      'title' => $this->t('Fixing Redirect Chains'),
      'operations' => [
        [[RedirectAuditBatch::class, 'processFix'], []],
      ],
      'finished' => [RedirectAuditBatch::class, 'finishedFix'],
    ];

    batch_set($batch);
  }

  /**
   * Submit handler for clear button.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function clearSubmit(array &$form, FormStateInterface $form_state): void {
    $count = $this->storage->clearProcessed();
    $this->messenger->addStatus($this->t('Removed @count records', ['@count' => $count]));
  }

  /**
   * AJAX callback for autofix toggle.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form element to update.
   */
  public function ajaxAutofixToggle(array &$form, FormStateInterface $form_state): array {
    $this->configFactory->getEditable('redirect_audit.settings')
      ->set('autofix_enabled', (bool) $form_state->getValue('autofix_enabled'))
      ->save();

    return $form['autofix_enabled'];
  }

}
