<?php

namespace Drupal\redirect_audit;

/**
 * Batch operations.
 */
class RedirectAuditBatch {

  /**
   * Run audit analysis.
   */
  public static function processAudit(array &$context): void {
    $context['message'] = t('Analyzing redirects for chains and loops...');
    $context['results'] = \Drupal::service('redirect_audit.analyzer')->detectChains();
    $context['finished'] = 1;
  }

  /**
   * Batch finished callback for audit.
   */
  public static function finishedAudit(bool $success, array $results, array $operations): void {
    if ($success) {
      $chains = ($results['chains_found'] ?? 0) - ($results['loops_found'] ?? 0);
      \Drupal::messenger()->addStatus(t(
        'Scan completed: @analyzed redirects analyzed. Found @chains chains and @loops loops.',
        [
          '@analyzed' => $results['analyzed'] ?? 0,
          '@chains' => $chains,
          '@loops' => $results['loops_found'] ?? 0,
        ]
      ));
    }
    else {
      \Drupal::messenger()->addError(t('An error occurred during the audit process.'));
    }
  }

  /**
   * Run fix for all chains.
   */
  public static function processFix(array &$context): void {
    $context['message'] = t('Fixing detected chains...');
    $context['results'] = \Drupal::service('redirect_audit.fixer')->fixAll();
    $context['finished'] = 1;
  }

  /**
   * Batch finished callback for fix.
   */
  public static function finishedFix(bool $success, array $results, array $operations): void {
    if ($success) {
      if (($results['fixed'] ?? 0) > 0) {
        \Drupal::messenger()->addStatus(t('Fixed @fixed chains', ['@fixed' => $results['fixed']]));
      }
      if (($results['errors'] ?? 0) > 0) {
        \Drupal::messenger()->addWarning(t('Could not fix @errors redirects', ['@errors' => $results['errors']]));
      }
      if (($results['skipped'] ?? 0) > 0) {
        \Drupal::messenger()->addWarning(t('Skipped @skipped loops (remove manually)', ['@skipped' => $results['skipped']]));
      }
    }
    else {
      \Drupal::messenger()->addError(t('An error occurred during the fix process.'));
    }
  }

}
