<?php

namespace Drupal\redirect_audit\Service;

use Drupal\Core\Database\Connection;

/**
 * Service handles all database operations for the redirect_audit_chains.
 */
class RedirectAuditStorage {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * RedirectAuditStorage object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * Saves a detected chain to the database.
   *
   * @param int $source_rid
   *   The source redirect ID.
   * @param int $target_rid
   *   The target redirect ID.
   * @param string $path
   *   The intermediate path.
   *
   * @return int
   *   The ID of the inserted record, or the existing record ID if duplicate.
   *
   * @throws \InvalidArgumentException
   *   If source_rid or target_rid are not valid.
   */
  public function saveChain(int $source_rid, int $target_rid, string $path): int {
    if ($source_rid <= 0 || $target_rid <= 0) {
      throw new \InvalidArgumentException('Source and target redirect IDs must be greater than 0.');
    }

    // Check if this chain already exists.
    $existing = $this->chainExists($source_rid, $target_rid, $path);
    if ($existing) {
      return (int) $existing['id'];
    }

    // Insert the chain record into the database.
    $id = $this->database->insert('redirect_audit_chains')
      ->fields([
        'source_rid' => $source_rid,
        'target_rid' => $target_rid,
        'path' => $path,
      ])
      ->execute();

    return (int) $id;
  }

  /**
   * Checks if a chain already exists in the database.
   *
   * @param int $source_rid
   *   The source redirect ID.
   * @param int $target_rid
   *   The target redirect ID.
   * @param string $path
   *   The intermediate path.
   *
   * @return array|null
   *   The existing chain record or NULL if not found.
   */
  public function chainExists(int $source_rid, int $target_rid, string $path): ?array {
    return $this->database->select('redirect_audit_chains', 'rac')
      ->fields('rac')
      ->condition('source_rid', $source_rid)
      ->condition('target_rid', $target_rid)
      ->condition('path', $path)
      ->execute()
      ->fetchAssoc() ?: NULL;
  }

  /**
   * Retrieves all detected chains from the database.
   *
   * @return array
   *   An array of chain records.
   */
  public function getChains(): array {
    return $this->database->select('redirect_audit_chains', 'rac')
      ->fields('rac')
      ->orderBy('id', 'DESC')
      ->execute()
      ->fetchAll(\PDO::FETCH_ASSOC);
  }

  /**
   * Retrieves a single chain by ID.
   *
   * @param int $id
   *   The chain ID.
   *
   * @return array|null
   *   The chain record or NULL if not found.
   */
  public function getChainById(int $id): ?array {
    return $this->database->select('redirect_audit_chains', 'rac')
      ->fields('rac')
      ->condition('id', $id)
      ->execute()
      ->fetchAssoc() ?: NULL;
  }

  /**
   * Gets chain and loop statistics.
   *
   * @return array
   *   Array with: total, chains, loops.
   */
  public function getStats(): array {
    $query = $this->database->query(
      "SELECT
        COUNT(*) as total,
        SUM(CASE WHEN source_rid = target_rid THEN 1 ELSE 0 END) as loops,
        SUM(CASE WHEN source_rid != target_rid THEN 1 ELSE 0 END) as chains
      FROM {redirect_audit_chains}"
    );
    $result = $query->fetchAssoc();

    return [
      'total' => (int) ($result['total'] ?? 0),
      'chains' => (int) ($result['chains'] ?? 0),
      'loops' => (int) ($result['loops'] ?? 0),
    ];
  }

  /**
   * Counts the total number of chains in the database.
   *
   * @return int
   *   The number of chain records.
   */
  public function countChains(): int {
    return (int) $this->database->select('redirect_audit_chains')
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Deletes all chain records from the database.
   *
   * @return int
   *   The number of deleted records.
   */
  public function clearProcessed(): int {
    $count = $this->countChains();

    $this->database->delete('redirect_audit_chains')->execute();
    return $count;
  }

  /**
   * Deletes a specific chain by ID.
   *
   * @param int $id
   *   The chain ID to delete.
   */
  public function deleteChain(int $id): void {
    $this->database->delete('redirect_audit_chains')
      ->condition('id', $id)
      ->execute();
  }

  /**
   * Deletes all audit records where a redirect is source OR target.
   *
   * @param int $redirect_id
   *   The redirect entity ID.
   *
   * @return int
   *   The number of records deleted.
   */
  public function deleteByRedirectId(int $redirect_id): int {
    return $this->database->delete('redirect_audit_chains')
      ->condition(
        $this->database->condition('OR')
          ->condition('source_rid', $redirect_id)
          ->condition('target_rid', $redirect_id)
      )
      ->execute();
  }

}
