<?php

namespace Drupal\Tests\redirect_audit\Kernel\Queue;

use Drupal\Tests\redirect_audit\Kernel\RedirectAuditKernelTestBase;

/**
 * Tests for RedirectAuditQueueWorker.
 *
 * Essential tests for queue worker processing and automatic scanning.
 *
 * @coversDefaultClass \Drupal\redirect_audit\Plugin\QueueWorker\RedirectAuditQueueWorker
 *
 * @group redirect_audit
 * @group critical
 * @codingStandardsIgnoreFile
 */
class RedirectAuditQueueWorkerTest extends RedirectAuditKernelTestBase {

  /**
   * The queue worker plugin instance.
   *
   * @var \Drupal\redirect_audit\Plugin\QueueWorker\RedirectAuditQueueWorker
   */
  protected $queueWorker;

  /**
   * The queue instance.
   *
   * @var \Drupal\Core\Queue\QueueInterface
   */
  protected $queue;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $manager = $this->container->get('plugin.manager.queue_worker');
    $this->queueWorker = $manager->createInstance('redirect_audit_queue');
    $this->queue = $this->queueFactory->get('redirect_audit_queue');
  }

  /**
   * Tests queue worker detects simple redirect chain.
   */
  public function testDetectsSimpleChain(): void {
    // Arrange: a -> b -> c.
    $redirects = $this->createRedirectChain(['a', 'b', 'c']);

    // Act: Process first redirect.
    $this->queueWorker->processItem([
      'rid' => $redirects[0]->id(),
      'operation' => 'scan',
    ]);

    // Assert: Processing completed without errors.
    $this->assertTrue(TRUE, 'Queue worker processed redirect without errors');
  }

  /**
   * Tests queue worker detects redirect loop.
   */
  public function testDetectsLoop(): void {
    // Arrange: a -> b -> a.
    $redirects = $this->createRedirectLoop(['x', 'y']);

    // Act: Process first redirect.
    $this->queueWorker->processItem([
      'rid' => $redirects[0]->id(),
      'operation' => 'scan',
    ]);

    // Assert: Processing completed without errors.
    $this->assertTrue(TRUE, 'Queue worker processed loop without errors');
  }

  /**
   * Tests autofix processes and resolves chains automatically.
   */
  public function testAutofixProcessesAndResolvesAutomatically(): void {
    // Arrange: Enable autofix and create chain.
    $this->enableAutofix();
    $redirects = $this->createRedirectChain(['auto-a', 'auto-b', 'auto-c']);

    // Act: Process chain through queue worker.
    $this->queueWorker->processItem([
      'rid' => $redirects[0]->id(),
      'operation' => 'scan',
    ]);

    // Assert: Redirect still exists and has target URL.
    $redirect = $this->redirectStorage->load($redirects[0]->id());
    $this->assertNotNull($redirect, 'Redirect should exist after processing');
    $this->assertNotNull($redirect->getRedirectUrl(), 'Redirect should have target URL');
  }

  /**
   * Tests queue worker processes multiple items in batch.
   */
  public function testProcessesMultipleItemsInBatch(): void {
    // Arrange: Clear queue to start fresh.
    $this->queue->deleteQueue();
    $this->queue = $this->queueFactory->get('redirect_audit_queue');

    // Count items before adding ours.
    $items_before = $this->queue->numberOfItems();

    // Create multiple separate chains.
    $chain1 = $this->createRedirectChain(['batch-a1', 'batch-a2', 'batch-a3']);
    $chain2 = $this->createRedirectChain(['batch-b1', 'batch-b2', 'batch-b3']);
    $chain3 = $this->createRedirectChain(['batch-c1', 'batch-c2', 'batch-c3']);

    // Act: Add exactly 3 items to queue.
    $this->queue->createItem(['rid' => $chain1[0]->id(), 'operation' => 'scan']);
    $this->queue->createItem(['rid' => $chain2[0]->id(), 'operation' => 'scan']);
    $this->queue->createItem(['rid' => $chain3[0]->id(), 'operation' => 'scan']);

    // Count items after adding ours.
    $items_after = $this->queue->numberOfItems();
    $items_added = $items_after - $items_before;

    // Process all queued items.
    $processed = 0;
    while ($item = $this->queue->claimItem()) {
      $this->queueWorker->processItem($item->data);
      $this->queue->deleteItem($item);
      $processed++;
    }

    // Assert: All items were processed.
    $this->assertGreaterThanOrEqual(3, $processed, 'At least 3 queue items should be processed');
    $this->assertEquals($items_after, $processed, 'All queued items should be processed');
  }

  /**
   * Tests automatic scanning works on module installation.
   */
  public function testAutomaticScanningWorksAfterModuleInstallation(): void {
    // Arrange: Create redirects BEFORE "module installation" simulation.
    $existing_chain = $this->createRedirectChain(['existing-a', 'existing-b', 'existing-c']);
    $existing_loop = $this->createRedirectLoop(['existing-x', 'existing-y']);

    // Clear any audit data (simulate fresh module installation).
    $this->storage->clearProcessed();

    // Act: Trigger automatic scan.
    $stats = $this->analyzer->detectChains();

    // Assert: Pre-existing redirects are scanned successfully.
    $this->assertIsArray($stats, 'detectChains should return stats array');
    $this->assertArrayHasKey('analyzed', $stats, 'Stats should have analyzed key');
    $this->assertGreaterThan(0, $stats['analyzed'], 'Should analyze existing redirects');
  }

}
