<?php

namespace Drupal\redirect_audit\Plugin\QueueWorker;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\redirect\Entity\Redirect;
use Drupal\redirect_audit\Service\RedirectAuditAnalyzer;
use Drupal\redirect_audit\Service\RedirectAuditStorage;
use Drupal\redirect_audit\Service\RedirectAuditFixer;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes redirect audit queue items.
 *
 * @QueueWorker(
 *   id = "redirect_audit_queue",
 *   title = @Translation("Redirect Audit Queue Worker"),
 *   cron = {"time" = 60}
 * )
 */
class RedirectAuditQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * The redirect audit analyzer service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditAnalyzer
   */
  protected RedirectAuditAnalyzer $analyzer;

  /**
   * The redirect audit storage service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditStorage
   */
  protected RedirectAuditStorage $storage;

  /**
   * The redirect audit fixer service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditFixer
   */
  protected RedirectAuditFixer $fixer;

  /**
   * The redirect entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $redirectStorage;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Whether autofix is enabled (cached).
   *
   * @var bool|null
   */
  protected ?bool $autofixEnabled = NULL;

  /**
   * Constructs a RedirectAuditQueueWorker object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\redirect_audit\Service\RedirectAuditAnalyzer $analyzer
   *   The redirect audit analyzer service.
   * @param \Drupal\redirect_audit\Service\RedirectAuditStorage $storage
   *   The redirect audit storage service.
   * @param \Drupal\redirect_audit\Service\RedirectAuditFixer $fixer
   *   The redirect audit fixer service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    RedirectAuditAnalyzer $analyzer,
    RedirectAuditStorage $storage,
    RedirectAuditFixer $fixer,
    EntityTypeManagerInterface $entity_type_manager,
    LoggerInterface $logger,
    ConfigFactoryInterface $config_factory,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->analyzer = $analyzer;
    $this->storage = $storage;
    $this->fixer = $fixer;
    $this->redirectStorage = $entity_type_manager->getStorage('redirect');
    $this->logger = $logger;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('redirect_audit.analyzer'),
      $container->get('redirect_audit.storage'),
      $container->get('redirect_audit.fixer'),
      $container->get('entity_type.manager'),
      $container->get('logger.factory')->get('redirect_audit'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    if (!$this->validateQueueData($data)) {
      return;
    }

    $rid = (int) $data['rid'];

    try {
      $redirect = $this->redirectStorage->load($rid);

      if (!$redirect) {
        $this->logger->debug('Redirect @rid no longer exists, skipping queue item', ['@rid' => $rid]);
        return;
      }

      $this->logger->debug('Processing redirect @rid (operation: @op, autofix: @autofix)', [
        '@rid' => $rid,
        '@op' => $data['operation'],
        '@autofix' => $this->isAutofixEnabled() ? 'enabled' : 'disabled',
      ]);

      $this->analyzeAndProcessRedirect($redirect, $rid);

      $this->logger->debug('Successfully processed redirect @rid', ['@rid' => $rid]);
    }
    catch (\Exception $e) {
      $this->logger->error('Error processing redirect @rid in queue: @message. Trace: @trace', [
        '@rid' => $rid,
        '@message' => $e->getMessage(),
        '@trace' => $e->getTraceAsString(),
      ]);

      throw $e;
    }
  }

  /**
   * Validates queue item data.
   *
   * @param mixed $data
   *   The queue item data.
   *
   * @return bool
   *   TRUE if valid.
   */
  protected function validateQueueData($data): bool {
    // Check required fields.
    if (!isset($data['rid']) || !isset($data['operation'])) {
      $this->logger->error('Invalid queue item: missing rid or operation');
      return FALSE;
    }

    $rid = $data['rid'];

    if (!is_numeric($rid) || $rid <= 0) {
      $this->logger->error('Invalid queue item: rid must be a positive integer, got @rid', [
        '@rid' => var_export($rid, TRUE),
      ]);
      return FALSE;
    }

    $data['rid'] = (int) $rid;

    return TRUE;
  }

  /**
   * Gets whether autofix is enabled (cached).
   *
   * @return bool
   *   TRUE if autofix is enabled.
   */
  protected function isAutofixEnabled(): bool {
    if ($this->autofixEnabled === NULL) {
      $config = $this->configFactory->get('redirect_audit.settings');
      $this->autofixEnabled = $config->get('autofix_enabled') ?? FALSE;
    }
    return $this->autofixEnabled;
  }

  /**
   * Analyzes a redirect and processes any detected chains.
   *
   * @param \Drupal\redirect\Entity\Redirect $redirect
   *   The redirect entity to analyze.
   * @param int $rid
   *   The redirect ID (for logging).
   */
  protected function analyzeAndProcessRedirect(Redirect $redirect, int $rid): void {
    // Analyze the redirect itself to detect if it forms a chain or loop.
    $chain_data = $this->analyzer->analyzeRedirect($redirect);

    if ($chain_data) {
      $is_loop = ($chain_data['source_rid'] === $chain_data['target_rid']);

      if ($is_loop) {
        // Check if this loop already exists to avoid duplicates.
        if (!$this->analyzer->loopAlreadyExists($chain_data)) {
          $this->processChainData($chain_data, $rid, FALSE);
        }
        else {
          $this->logger->debug('Loop already exists for redirect @rid, skipping duplicate', ['@rid' => $rid]);
        }
      }
      elseif (!$this->analyzer->isIntermediateRedirect($redirect)) {
        $this->processChainData($chain_data, $rid, FALSE);
      }
      else {
        $this->logger->debug('Skipping chain from intermediate redirect @rid', ['@rid' => $rid]);
      }
    }

  }

  /**
   * Processes and saves chain data.
   *
   * @param array $chain_data
   *   The chain data array with source_rid, target_rid, and path.
   * @param int $rid
   *   The redirect ID being processed (for logging).
   * @param bool $is_pointing_chain
   *   Whether this is a chain pointing to the redirect (for logging).
   */
  protected function processChainData(array $chain_data, int $rid, bool $is_pointing_chain): void {
    // Validate chain data structure.
    if (!$this->isValidChainData($chain_data)) {
      $this->logger->error('Invalid chain data for redirect @rid: missing required fields', [
        '@rid' => $rid,
      ]);
      return;
    }

    try {
      // Save the chain to the audit table.
      $audit_id = $this->storage->saveChain(
        $chain_data['source_rid'],
        $chain_data['target_rid'],
        $chain_data['path']
      );

      $is_loop = ($chain_data['source_rid'] === $chain_data['target_rid']);
      $this->logChainDetection($rid, $audit_id, $is_loop, $is_pointing_chain);

      // Attempt auto-fix if enabled.
      if ($this->isAutofixEnabled()) {
        if (!$is_loop) {
          $this->attemptAutoFix($audit_id);
        }
        else {
          $this->logLoopNoAutofix($audit_id);
        }
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to save chain data for redirect @rid: @message', [
        '@rid' => $rid,
        '@message' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Validates chain data structure.
   *
   * @param array $chain_data
   *   The chain data to validate.
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  protected function isValidChainData(array $chain_data): bool {
    return isset($chain_data['source_rid'], $chain_data['target_rid'], $chain_data['path']);
  }

  /**
   * Logs chain detection.
   *
   * @param int $rid
   *   The redirect ID being processed.
   * @param int $audit_id
   *   The audit record ID.
   * @param bool $is_loop
   *   Whether this is a loop.
   * @param bool $is_pointing_chain
   *   Whether this chain points to the redirect.
   */
  protected function logChainDetection(int $rid, int $audit_id, bool $is_loop, bool $is_pointing_chain): void {
    $type = $is_loop ? 'loop' : 'chain';

    if ($is_pointing_chain) {
      $this->logger->info('Chain detected pointing to redirect @rid: audit ID @audit_id, type: @type', [
        '@rid' => $rid,
        '@audit_id' => $audit_id,
        '@type' => $type,
      ]);
    }
    else {
      $this->logger->info('Chain detected for redirect @rid: audit ID @audit_id, type: @type', [
        '@rid' => $rid,
        '@audit_id' => $audit_id,
        '@type' => $type,
      ]);
    }
  }

  /**
   * Attempts to auto-fix a chain.
   *
   * @param int $audit_id
   *   The audit record ID.
   */
  protected function attemptAutoFix(int $audit_id): void {
    try {
      $this->fixer->fixChain($audit_id);
      $this->logger->info('Chain auto-fixed (queue): audit ID @id', ['@id' => $audit_id]);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to auto-fix chain @id: @message', [
        '@id' => $audit_id,
        '@message' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Logs that a loop cannot be auto-fixed.
   *
   * @param int $audit_id
   *   The audit record ID.
   */
  protected function logLoopNoAutofix(int $audit_id): void {
    $this->logger->notice('Loop detected (audit ID @id) - cannot auto-fix, manual intervention required', [
      '@id' => $audit_id,
    ]);
  }

}
