<?php

namespace Drupal\redirect_audit;

/**
 * Batch operations.
 */
class RedirectAuditBatch {

  /**
   * Run audit analysis.
   */
  public static function processAudit(array &$context): void {
    $context['message'] = t('Analyzing redirects for chains and loops...');
    $context['results'] = \Drupal::service('redirect_audit.analyzer')->detectChains();
    $context['finished'] = 1;
  }

  /**
   * Batch finished callback for audit.
   */
  public static function finishedAudit(bool $success, array $results, array $operations): void {
    if ($success) {
      $chains = ($results['chains_found'] ?? 0) - ($results['loops_found'] ?? 0);
      $loops = $results['loops_found'] ?? 0;

      if ($chains > 0 || $loops > 0) {
        \Drupal::messenger()->addStatus(t('Analysis complete: found @chains chains and @loops loops', [
          '@chains' => $chains,
          '@loops' => $loops,
        ]));
      }
      else {
        \Drupal::messenger()->addStatus(t('Analysis complete: no issues found'));
      }
    }
    else {
      \Drupal::messenger()->addError(t('Analysis failed. Please check the logs.'));
    }
  }

  /**
   * Run fix for all chains.
   */
  public static function processFix(array &$context): void {
    $context['message'] = t('Fixing detected chains...');
    $context['results'] = \Drupal::service('redirect_audit.fixer')->fixAll();
    $context['finished'] = 1;
  }

  /**
   * Batch finished callback for fix.
   */
  public static function finishedFix(bool $success, array $results, array $operations): void {
    if ($success) {
      $fixed = $results['fixed'] ?? 0;
      $errors = $results['errors'] ?? 0;
      $skipped = $results['skipped'] ?? 0;

      if ($fixed > 0) {
        \Drupal::messenger()->addStatus(t('Successfully fixed @count chains', [
          '@count' => $fixed,
        ]));
      }

      if ($errors > 0) {
        \Drupal::messenger()->addWarning(t('@count chains could not be fixed', [
          '@count' => $errors,
        ]));
      }

      if ($skipped > 0) {
        \Drupal::messenger()->addWarning(t('@count loops require manual correction', [
          '@count' => $skipped,
        ]));
      }

      if ($fixed === 0 && $errors === 0 && $skipped === 0) {
        \Drupal::messenger()->addStatus(t('No chains to fix'));
      }
    }
    else {
      \Drupal::messenger()->addError(t('Fix operation failed. Please check the logs.'));
    }
  }

}
