<?php

namespace Drupal\redirect_audit;

use Drupal\redirect\Entity\Redirect;

/**
 * Interface for redirect chain resolver service.
 */
interface RedirectChainResolverInterface {

  /**
   * Mark redirect object to process.
   *
   * @param \Drupal\redirect\Entity\Redirect $redirect
   *   Redirect object.
   */
  public function markRedirectObjectAsUnprocessed(Redirect $redirect);

  /**
   * Marks redirects as unprocessed using direct database update.
   *
   * @param int[] $ids
   *   Array of redirect IDs to mark as unprocessed.
   */
  public function markRedirectsAsUnprocessed(array $ids);

  /**
   * Marks all redirects as unprocessed.
   *
   * @return int
   *   Number of redirects marked as unprocessed.
   */
  public function markAllAsUnprocessed();

  /**
   * Marks redirects to source for processing.
   *
   * @param \Drupal\redirect\Entity\Redirect $redirect
   *   Redirect object.
   */
  public function findAndMarkRedirectsToSource(Redirect $redirect);

  /**
   * Marks redirects pointing to a specific path as unprocessed.
   *
   * @param string $path
   *   The path to search for.
   */
  public function findAndMarkRedirectsTo($path);

  /**
   * Processes a single redirect and marks it as processed.
   *
   * @param \Drupal\redirect\Entity\Redirect $redirect
   *   The redirect to process.
   *
   * @return bool
   *   TRUE if the redirect was updated (chain resolved), FALSE otherwise.
   */
  public function processRedirect(Redirect $redirect);

  /**
   * Processes a batch of unprocessed redirects (single pass).
   *
   * @param int $limit
   *   Maximum number of redirects to process.
   *
   * @return array
   *   Array with stats: processed, updated, remaining.
   */
  public function processRedirects($limit = 50);

  /**
   * Processes a batch of unprocessed redirects with multiple passes.
   *
   * @param int $batch_size
   *   Number of redirects to process.
   * @param int $max_iterations
   *   Maximum number of iterations to prevent infinite loops.
   *
   * @return array
   *   Array with 'processed', 'updated', 'iterations', and 'remaining' counts.
   */
  public function processBatch($batch_size = 50, $max_iterations = 10);

  /**
   * Counts unprocessed redirects.
   *
   * @return int
   *   Number of unprocessed redirects.
   */
  public function countUnprocessedRedirects();

  /**
   * Gets unprocessed redirects for batch processing.
   *
   * @param int $limit
   *   Maximum number of redirects to return.
   * @param int $offset
   *   Offset for pagination.
   *
   * @return \Drupal\redirect\Entity\Redirect[]
   *   Array of unprocessed redirect entities.
   */
  public function getUnprocessedRedirects($limit = 50, $offset = 0);

  /**
   * Checks if a redirect is part of a chain and resolves it.
   *
   * @param \Drupal\redirect\Entity\Redirect $redirect
   *   The redirect to check and resolve.
   * @param int $max_depth
   *   Maximum chain depth to prevent infinite loops.
   *
   * @return bool
   *   TRUE if the redirect was updated, FALSE otherwise.
   */
  public function checkAndResolveChain(Redirect $redirect, $max_depth = 10);

  /**
   * Marks a redirect as processed.
   *
   * @param \Drupal\redirect\Entity\Redirect $redirect
   *   The redirect entity to mark as processed.
   */
  public function markAsProcessed(Redirect $redirect);

}
