<?php

namespace Drupal\redirect_audit\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\redirect_audit\RedirectChainResolverInterface;

/**
 * Service for fixing redirect chains and removing loops.
 */
class RedirectAuditFixer {

  /**
   * The redirect chain resolver service.
   *
   * @var \Drupal\redirect_audit\RedirectChainResolverInterface
   */
  protected RedirectChainResolverInterface $chainResolver;

  /**
   * The redirect audit storage service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditStorage
   */
  protected RedirectAuditStorage $storage;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs a RedirectAuditFixer object.
   *
   * @param \Drupal\redirect_audit\RedirectChainResolverInterface $chain_resolver
   *   The redirect chain resolver service.
   * @param \Drupal\redirect_audit\Service\RedirectAuditStorage $storage
   *   The redirect audit storage service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   */
  public function __construct(
    RedirectChainResolverInterface $chain_resolver,
    RedirectAuditStorage $storage,
    EntityTypeManagerInterface $entity_type_manager,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $this->chainResolver = $chain_resolver;
    $this->storage = $storage;
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger_factory->get('redirect_audit');
  }

  /**
   * Fixes a specific redirect chain.
   *
   * @param int $audit_id
   *   The audit record ID.
   *
   * @return bool
   *   TRUE if the chain was fixed, FALSE otherwise.
   */
  public function fixChain(int $audit_id): bool {
    try {
      $record = $this->storage->getChainById($audit_id);
      if (!$record) {
        $this->logger->warning('Audit record @id not found', ['@id' => $audit_id]);
        return FALSE;
      }

      $redirect_storage = $this->entityTypeManager->getStorage('redirect');

      // Load the source redirect.
      $redirect = $redirect_storage->load($record['source_rid']);
      if (!$redirect) {
        $this->logger->warning('Redirect @rid no longer exists, removing from audit', [
          '@rid' => $record['source_rid'],
        ]);
        $this->storage->deleteChain($audit_id);
        return FALSE;
      }

      // Extract intermediate redirect IDs.
      $intermediate_ids = array_filter(array_map('intval', explode('.', $record['path'])));
      $all_ids_to_fix = array_merge([$record['source_rid']], $intermediate_ids);

      // Fix all redirects in the chain.
      $fixed_count = 0;
      foreach ($all_ids_to_fix as $rid) {
        $redirect_to_fix = $redirect_storage->load($rid);
        if ($redirect_to_fix) {
          $fixed = $this->chainResolver->checkAndResolveChain($redirect_to_fix);
          if ($fixed) {
            $fixed_count++;
          }
        }
      }

      if ($fixed_count > 0) {
        $this->storage->deleteChain($audit_id);
        $this->logger->info('Chain fixed: updated @count redirect(s)', [
          '@count' => $fixed_count,
        ]);
        return TRUE;
      }

      $this->logger->warning('Unable to fix chain (redirects may have changed)');
      return FALSE;
    }
    catch (\Exception $e) {
      $this->logger->error('Error fixing chain @id: @message', [
        '@id' => $audit_id,
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Fixes all detected redirect chains.
   *
   * @return array
   *   Stats: fixed, errors, skipped, total.
   */
  public function fixAll(): array {
    $fixed = 0;
    $errors = 0;
    $skipped = 0;

    $chains = $this->storage->getChains();
    $total = count($chains);

    $this->logger->info('Starting to fix @total chain(s)', [
      '@total' => $total,
    ]);

    foreach ($chains as $record) {
      // Skip loops.
      if ($record['source_rid'] === $record['target_rid']) {
        $skipped++;
        continue;
      }

      // Fix the chain.
      $result = $this->fixChain($record['id']);
      if ($result) {
        $fixed++;
      }
      else {
        $errors++;
      }

      // Log progress every 10 records.
      $processed = $fixed + $errors + $skipped;
      if ($processed % 10 === 0) {
        $this->logger->info('Progress: @processed/@total processed', [
          '@processed' => $processed,
          '@total' => $total,
        ]);
      }
    }

    $this->logger->info('Fix complete: @fixed fixed, @errors failed, @skipped loops skipped', [
      '@fixed' => $fixed,
      '@errors' => $errors,
      '@skipped' => $skipped,
    ]);

    return [
      'fixed' => $fixed,
      'errors' => $errors,
      'skipped' => $skipped,
      'total' => $total,
    ];
  }

}
