<?php

namespace Drupal\Tests\redirect_audit\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\redirect\Entity\Redirect;

/**
 * Base class for Redirect Audit kernel tests.
 *
 * Provides common functionality for kernel tests including module installation,
 * redirect entity setup, and shared configuration.
 *
 * @codingStandardsIgnoreFile
 */
abstract class RedirectAuditKernelTestBase extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'path_alias',
    'link',
    'views',
    'redirect',
    'redirect_audit',
  ];

  /**
   * The redirect audit analyzer service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditAnalyzer
   */
  protected $analyzer;

  /**
   * The redirect audit storage service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditStorage
   */
  protected $storage;

  /**
   * The redirect audit fixer service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditFixer
   */
  protected $fixer;

  /**
   * The redirect entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $redirectStorage;

  /**
   * The queue service.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queueFactory;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install necessary schemas.
    $this->installEntitySchema('redirect');
    $this->installEntitySchema('path_alias');
    $this->installSchema('redirect_audit', ['redirect_audit_chains', 'redirect_audit_processed']);

    // Install configuration.
    $this->installConfig(['system', 'redirect', 'redirect_audit']);

    // Get services.
    $this->analyzer = $this->container->get('redirect_audit.analyzer');
    $this->storage = $this->container->get('redirect_audit.storage');
    $this->fixer = $this->container->get('redirect_audit.fixer');
    $this->redirectStorage = $this->container->get('entity_type.manager')->getStorage('redirect');
    $this->queueFactory = $this->container->get('queue');

    // Set default configuration.
    $this->config('redirect_audit.settings')
      ->set('autofix_enabled', FALSE)
      ->set('max_chain_depth', 10)
      ->save();
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    // Clear all audit data.
    $this->storage->clearProcessed();

    parent::tearDown();
  }

  /**
   * Creates a test redirect.
   *
   * @param string $source_path
   *   The source path (without leading slash).
   * @param string $redirect_path
   *   The redirect destination path.
   * @param int $status_code
   *   The redirect status code (default: 301).
   * @param string $language
   *   The language code (default: 'en').
   *
   * @return \Drupal\redirect\Entity\Redirect
   *   The created redirect entity.
   */
  protected function createRedirect(
    string $source_path,
    string $redirect_path,
    int $status_code = 301,
    string $language = 'en'
  ): Redirect {
    $redirect = Redirect::create([
      'redirect_source' => $source_path,
      'redirect_redirect' => 'internal:/' . ltrim($redirect_path, '/'),
      'language' => $language,
      'status_code' => $status_code,
    ]);
    $redirect->save();

    return $redirect;
  }

  /**
   * Creates a chain of redirects for testing.
   *
   * @param array $paths
   *   Array of paths where each redirects to the next.
   *   Example: ['path1', 'path2', 'path3'] creates:
   *   - path1 -> path2
   *   - path2 -> path3.
   *
   * @return array
   *   Array of created Redirect entities.
   */
  protected function createRedirectChain(array $paths): array {
    $redirects = [];
    $count = count($paths);

    for ($i = 0; $i < $count - 1; $i++) {
      $redirects[] = $this->createRedirect($paths[$i], $paths[$i + 1]);
    }

    return $redirects;
  }

  /**
   * Creates a redirect loop for testing.
   *
   * @param array $paths
   *   Array of paths that form a loop.
   *   Example: ['a', 'b', 'c'] creates: a -> b -> c -> a.
   *
   * @return array
   *   Array of created Redirect entities.
   */
  protected function createRedirectLoop(array $paths): array {
    $redirects = [];
    $count = count($paths);

    for ($i = 0; $i < $count; $i++) {
      $next_index = ($i + 1) % $count;
      $redirects[] = $this->createRedirect($paths[$i], $paths[$next_index]);
    }

    return $redirects;
  }

  /**
   * Enables autofix in configuration.
   */
  protected function enableAutofix(): void {
    $this->config('redirect_audit.settings')
      ->set('autofix_enabled', TRUE)
      ->save();
  }

  /**
   * Gets all items from a queue without removing them.
   *
   * @param string $queue_name
   *   The queue name.
   *
   * @return array
   *   Array of queue items.
   */
  protected function getQueueItems(string $queue_name): array {
    $queue = $this->queueFactory->get($queue_name);
    $items = [];

    // Claim and release items to inspect them.
    while ($item = $queue->claimItem()) {
      $items[] = $item->data;
      $queue->releaseItem($item);
    }

    return $items;
  }

  /**
   * Clears all items from a queue.
   *
   * @param string $queue_name
   *   The queue name.
   */
  protected function clearQueue(string $queue_name): void {
    $queue = $this->queueFactory->get($queue_name);
    $queue->deleteQueue();
  }

  /**
   * Asserts that a chain exists in the audit storage.
   *
   * @param int $source_rid
   *   The source redirect ID.
   * @param int $target_rid
   *   The target redirect ID.
   * @param string $message
   *   Optional assertion message.
   */
  protected function assertChainExists(int $source_rid, int $target_rid, string $message = ''): void {
    $chains = $this->storage->getChains();
    $found = FALSE;

    foreach ($chains as $chain) {
      if ($chain['source_rid'] == $source_rid && $chain['target_rid'] == $target_rid) {
        $found = TRUE;
        break;
      }
    }

    $this->assertTrue($found, $message ?: "Chain $source_rid -> $target_rid should exist");
  }

  /**
   * Asserts that a chain does not exist in the audit storage.
   *
   * @param int $source_rid
   *   The source redirect ID.
   * @param int $target_rid
   *   The target redirect ID.
   * @param string $message
   *   Optional assertion message.
   */
  protected function assertChainNotExists(int $source_rid, int $target_rid, string $message = ''): void {
    $chains = $this->storage->getChains();
    $found = FALSE;

    foreach ($chains as $chain) {
      if ($chain['source_rid'] == $source_rid && $chain['target_rid'] == $target_rid) {
        $found = TRUE;
        break;
      }
    }

    $this->assertFalse($found, $message ?: "Chain $source_rid -> $target_rid should not exist");
  }

}
