<?php

namespace Drupal\redirect_audit\Commands;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\redirect_audit\Service\RedirectAuditAnalyzer;
use Drupal\redirect_audit\Service\RedirectAuditFixer;
use Drupal\redirect_audit\Service\RedirectAuditStorage;
use Drush\Commands\DrushCommands;

/**
 * A Drush commands file.
 *
 * Provide Drush command for redirect_audit module.
 */
class RedirectAuditCommands extends DrushCommands {

  use StringTranslationTrait;

  /**
   * The redirect audit analyzer service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditAnalyzer
   */
  protected RedirectAuditAnalyzer $analyzer;

  /**
   * The redirect audit fixer service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditFixer
   */
  protected RedirectAuditFixer $fixer;

  /**
   * The redirect audit storage service.
   *
   * @var \Drupal\redirect_audit\Service\RedirectAuditStorage
   */
  protected RedirectAuditStorage $storage;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a RedirectAuditCommands object.
   *
   * @param \Drupal\redirect_audit\Service\RedirectAuditAnalyzer $analyzer
   *   The redirect audit analyzer service.
   * @param \Drupal\redirect_audit\Service\RedirectAuditFixer $fixer
   *   The redirect audit fixer service.
   * @param \Drupal\redirect_audit\Service\RedirectAuditStorage $storage
   *   The redirect audit storage service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    RedirectAuditAnalyzer $analyzer,
    RedirectAuditFixer $fixer,
    RedirectAuditStorage $storage,
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    parent::__construct();
    $this->analyzer = $analyzer;
    $this->fixer = $fixer;
    $this->storage = $storage;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Scan all redirects for chains and loops.
   *
   * @command redirect-audit:scan
   *
   * @usage drush redirect-audit:scan
   *   Scan all redirects for chains and loops.
   * @usage drush ras
   *   Scan using the alias.
   *
   * @aliases ras
   */
  public function scan(): int {
    try {
      $results = $this->analyzer->detectChains();

      $chains = ($results['chains_found'] ?? 0) - ($results['loops_found'] ?? 0);
      $loops = $results['loops_found'] ?? 0;

      $this->logger()->success($this->t('Scan complete.'));
      $this->logger()->notice($this->t('Found @chains chain(s) and @loops loop(s).', [
        '@chains' => $chains,
        '@loops' => $loops,
      ]));

      if ($chains > 0) {
        $this->logger()->notice($this->t('Run "drush raf" to fix chains.'));
      }
      if ($chains > 0 || $loops > 0) {
        $this->logger()->notice($this->t('Run "drush rai" for details.'));
      }

      return DrushCommands::EXIT_SUCCESS;
    }
    catch (\Exception $e) {
      $this->logger()->error($this->t('Error: @message', [
        '@message' => $e->getMessage(),
      ]));
      return DrushCommands::EXIT_FAILURE;
    }
  }

  /**
   * Fix all detected redirect chains.
   *
   * @command redirect-audit:fix
   *
   * @usage drush redirect-audit:fix
   *   Fix all redirect chains.
   * @usage drush raf
   *   Fix using the alias.
   *
   * @aliases raf
   */
  public function fix(): int {
    $stats = $this->storage->getStats();

    if ($stats['total'] === 0) {
      $this->logger()->warning($this->t('No issues found. Run "drush ras" first.'));
      return DrushCommands::EXIT_SUCCESS;
    }

    if ($stats['chains'] === 0) {
      if ($stats['loops'] > 0) {
        $this->logger()->warning($this->t('Only @count loop(s) found. Loops require manual correction.', [
          '@count' => $stats['loops'],
        ]));
      }
      else {
        $this->logger()->notice($this->t('No chains to fix.'));
      }
      return DrushCommands::EXIT_SUCCESS;
    }

    try {
      $results = $this->fixer->fixAll();

      $this->logger()->success($this->t('Fixed @count chain(s).', [
        '@count' => $results['fixed'] ?? 0,
      ]));

      if ($stats['loops'] > 0) {
        $this->logger()->warning($this->t('@count loop(s) require manual correction.', [
          '@count' => $stats['loops'],
        ]));
      }

      return DrushCommands::EXIT_SUCCESS;
    }
    catch (\Exception $e) {
      $this->logger()->error($this->t('Error: @message', [
        '@message' => $e->getMessage(),
      ]));
      return DrushCommands::EXIT_FAILURE;
    }
  }

  /**
   * Display audit information.
   *
   * @command redirect-audit:info
   *
   * @aliases rai
   */
  public function info(): int {
    $stats = $this->storage->getStats();
    $config = $this->configFactory->get('redirect_audit.settings');

    $this->io()->title($this->t('Redirect Audit'));

    if ($stats['total'] === 0) {
      $this->logger()->notice($this->t('No issues found.'));
      $this->logger()->notice($this->t('Run "drush ras" to scan.'));
      return DrushCommands::EXIT_SUCCESS;
    }

    // Statistics.
    $this->io()->section($this->t('Statistics'));
    $this->io()->table(
      [$this->t('Metric'), $this->t('Count')],
      [
        [$this->t('Total records'), $stats['total']],
        [$this->t('Redirect chains'), $stats['chains']],
        [$this->t('Redirect loops'), $stats['loops']],
      ]
    );

    // Configuration.
    $this->io()->section($this->t('Configuration'));
    $this->io()->table(
      [$this->t('Setting'), $this->t('Value')],
      [
        [
          $this->t('Autofix enabled'),
          $config->get('autofix_enabled') ? $this->t('Yes') : $this->t('No'),
        ],
        [
          $this->t('Scan on change'),
          $config->get('scan_on_change') ? $this->t('Yes') : $this->t('No'),
        ],
        [$this->t('Max chain depth'), $config->get('max_chain_depth') ?? 10],
        [$this->t('Batch size'), $config->get('batch_size') ?? 50],
      ]
    );

    // Detailed records.
    $this->io()->section($this->t('Detailed Records'));
    $chains = $this->storage->getChains(100);

    if (empty($chains)) {
      $this->logger()->notice($this->t('No records to display.'));
      return DrushCommands::EXIT_SUCCESS;
    }

    // Collect redirect IDs.
    $redirect_ids = [];
    foreach ($chains as $chain) {
      $redirect_ids[$chain['source_rid']] = $chain['source_rid'];
      $redirect_ids[$chain['target_rid']] = $chain['target_rid'];
      if (!empty($chain['path'])) {
        $intermediate_ids = array_filter(
          array_map('intval', explode('.', $chain['path']))
        );
        foreach ($intermediate_ids as $id) {
          $redirect_ids[$id] = $id;
        }
      }
    }

    // Load redirects.
    $redirect_storage = $this->entityTypeManager->getStorage('redirect');
    $redirects = $redirect_storage->loadMultiple($redirect_ids);

    // Build rows.
    $rows = [];
    foreach ($chains as $chain) {
      $is_loop = ($chain['source_rid'] === $chain['target_rid']);
      $type = $is_loop ? $this->t('Loop') : $this->t('Chain');

      $source_redirect = $redirects[$chain['source_rid']] ?? NULL;
      $source_path = $source_redirect ?
        $source_redirect->getSourcePathWithQuery() : $this->t('N/A');

      // Format intermediate redirect IDs.
      $intermediate_ids = '';
      if (!empty($chain['path'])) {
        $ids = array_filter(array_map('intval', explode('.', $chain['path'])));
        $intermediate_ids = implode(', ', $ids);
      }
      if (empty($intermediate_ids)) {
        $intermediate_ids = '-';
      }

      $target_redirect = $redirects[$chain['target_rid']] ?? NULL;
      $target_path = $this->t('N/A');
      if ($target_redirect && ($target_url = $target_redirect->getRedirectUrl())) {
        $target_path = $target_url->toString();
      }

      $rows[] = [
        $chain['id'],
        $type,
        $source_path,
        $intermediate_ids,
        $target_path,
      ];
    }

    $this->io()->table(
      [
        $this->t('ID'),
        $this->t('Type'),
        $this->t('Source Path'),
        $this->t('Intermediate Links'),
        $this->t('Target Path'),
      ],
      $rows
    );

    $total = $this->storage->countChains();
    if ($total > 100) {
      $this->logger()->notice($this->t('Showing first 100 of @total records.', [
        '@total' => $total,
      ]));
    }

    return DrushCommands::EXIT_SUCCESS;
  }

}
