<?php

namespace Drupal\redirect_audit;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\redirect\Entity\Redirect;
use Drupal\redirect\Exception\RedirectLoopException;

/**
 * Service to detect and resolve redirect chains.
 */
class RedirectChainResolver implements RedirectChainResolverInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Cached redirect storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|null
   */
  protected $redirectStorage;

  /**
   * Cached logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|null
   */
  protected $logger;

  /**
   * Constructs a RedirectChainResolver object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    LoggerChannelFactoryInterface $logger_factory,
    Connection $database,
    TimeInterface $time,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->loggerFactory = $logger_factory;
    $this->database = $database;
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public function markRedirectObjectAsUnprocessed(Redirect $redirect) {
    $this->markRedirectsAsUnprocessed([(int) $redirect->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function markRedirectsAsUnprocessed(array $ids) {
    if (empty($ids)) {
      return;
    }

    foreach ($ids as $rid) {
      $this->database->merge('redirect_audit_processed')
        ->key(['rid' => $rid])
        ->fields(['chain_processed' => 0])
        ->execute();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function markAllAsUnprocessed() {
    $rids = $this->database->select('redirect', 'r')
      ->fields('r', ['rid'])
      ->execute()
      ->fetchCol();

    if (empty($rids)) {
      return 0;
    }

    $this->markRedirectsAsUnprocessed($rids);
    return count($rids);
  }

  /**
   * {@inheritdoc}
   */
  public function findAndMarkRedirectsToSource(Redirect $redirect) {
    $source = $redirect->getSource();
    if (!empty($source['path'])) {
      $this->findAndMarkRedirectsTo($source['path']);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function findAndMarkRedirectsTo($path) {
    if (!$path) {
      return;
    }

    $rids = $this->database->select('redirect', 'r')
      ->fields('r', ['rid'])
      ->condition('redirect_redirect__uri', 'internal:/' . $path)
      ->execute()
      ->fetchCol();

    if (!empty($rids)) {
      $this->markRedirectsAsUnprocessed($rids);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function processRedirect(Redirect $redirect) {
    $was_updated = FALSE;
    try {
      $was_updated = $this->checkAndResolveChain($redirect);
    }
    catch (\Exception $e) {
      $this->getLogger()->error('Error processing redirect @rid: @message', [
        '@rid' => $redirect->id(),
        '@message' => $e->getMessage(),
      ]);
      $was_updated = FALSE;
    }
    $this->markAsProcessed($redirect);
    return $was_updated;
  }

  /**
   * {@inheritdoc}
   */
  public function processRedirects($limit = 50) {
    $redirects = $this->getUnprocessedRedirects($limit, 0);
    $processed = 0;
    $updated = 0;

    foreach ($redirects as $redirect) {
      if ($this->processRedirect($redirect)) {
        $updated++;
      }
      $processed++;
    }

    $remaining = $this->countUnprocessedRedirects();

    return [
      'processed' => $processed,
      'updated' => $updated,
      'remaining' => $remaining,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function processBatch($batch_size = 50, $max_iterations = 10) {
    $total_processed = 0;
    $total_updated = 0;
    $iteration = 0;
    $initial_unprocessed = $this->countUnprocessedRedirects();
    $remaining = $initial_unprocessed;

    // Multiple iterations to process all possible redirects.
    while ($remaining > 0 && $iteration < $max_iterations) {
      $iteration++;
      $redirects = $this->getUnprocessedRedirects($batch_size);

      if (empty($redirects)) {
        break;
      }

      $iteration_processed = 0;
      $iteration_updated = 0;

      foreach ($redirects as $redirect) {
        if ($this->processRedirect($redirect)) {
          $iteration_updated++;
        }
        $iteration_processed++;
      }

      $total_processed += $iteration_processed;
      $total_updated += $iteration_updated;

      $this->getLogger()->info('Chain processing iteration @iteration: processed @processed, updated @updated', [
        '@iteration' => $iteration,
        '@processed' => $iteration_processed,
        '@updated' => $iteration_updated,
      ]);

      $remaining = $this->countUnprocessedRedirects();
    }

    $final_remaining = $this->countUnprocessedRedirects();

    return [
      'processed' => $total_processed,
      'updated' => $total_updated,
      'iterations' => $iteration,
      'initial_unprocessed' => $initial_unprocessed,
      'remaining' => $final_remaining,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function countUnprocessedRedirects() {
    return (int) $this->buildUnprocessedQuery()->countQuery()->execute()->fetchField();
  }

  /**
   * {@inheritdoc}
   */
  public function getUnprocessedRedirects($limit = 50, $offset = 0) {
    $ids = $this->buildUnprocessedQuery()
      ->fields('r', ['rid'])
      ->range($offset, $limit)
      ->execute()
      ->fetchCol();

    return $this->getRedirectStorage()->loadMultiple($ids);
  }

  /**
   * Builds base query for unprocessed redirects.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface
   *   The query object.
   */
  protected function buildUnprocessedQuery() {
    $query = $this->database->select('redirect', 'r');
    $query->leftJoin('redirect_audit_processed', 'rap', 'r.rid = rap.rid');
    $query->condition(
      $query->orConditionGroup()
        ->isNull('rap.rid')
        ->condition('rap.chain_processed', 0)
    );

    return $query;
  }

  /**
   * Gets the redirect storage.
   *
   * @return \Drupal\Core\Entity\EntityStorageInterface
   *   The redirect storage.
   */
  protected function getRedirectStorage() {
    if (!$this->redirectStorage) {
      $this->redirectStorage = $this->entityTypeManager->getStorage('redirect');
    }
    return $this->redirectStorage;
  }

  /**
   * {@inheritdoc}
   */
  public function checkAndResolveChain(Redirect $redirect, $max_depth = 10) {
    try {
      $visited = [];
      // Follow the chain recursively to find all redirects.
      $chain = $this->followChain($redirect, $visited, 0, $max_depth);

      if (count($chain) > 1) {
        $final_redirect = end($chain);
        $final_destination_data = $final_redirect->getRedirect();

        $redirect->set('redirect_redirect', $final_destination_data);
        $redirect->save();

        return TRUE;
      }
    }
    catch (RedirectLoopException $e) {
      $this->getLogger()->error('Redirect loop detected: @message', ['@message' => $e->getMessage()]);
      throw $e;
    }

    return FALSE;
  }

  /**
   * Follows a redirect chain to find all redirects in the chain.
   *
   * @param \Drupal\redirect\Entity\Redirect $redirect
   *   The starting redirect.
   * @param array $visited
   *   Array to track visited redirects.
   * @param int $depth
   *   Current depth.
   * @param int $max_depth
   *   Maximum allowed depth.
   *
   * @return array
   *   Array of Redirect entities in the chain.
   *
   * @throws \Drupal\redirect\Exception\RedirectLoopException
   */
  protected function followChain(Redirect $redirect, array &$visited, $depth, $max_depth) {
    $rid = $redirect->id();

    // Loop detection.
    if (isset($visited[$rid])) {
      throw new RedirectLoopException($redirect->getSourcePathWithQuery(), (int) $rid);
    }

    if ($depth >= $max_depth) {
      return [$redirect];
    }

    $visited[$rid] = TRUE;
    $chain = [$redirect];
    $destination_url = $redirect->getRedirectUrl();

    if ($destination_url) {
      $path = '';
      $query = [];

      // Extract path and query from the destination URL.
      if ($destination_url->isRouted()) {
        $path = $destination_url->getInternalPath();
        $query = $destination_url->getOption('query') ?: [];
      }
      else {
        $uri_string = $destination_url->toString();
        $path = ltrim($uri_string, '/');
      }

      if ($path) {
        $next_redirect = $this->findRedirectByPath($path, $query, $redirect->language->value);

        if ($next_redirect && $next_redirect->id() != $rid) {
          $next_chain = $this->followChain($next_redirect, $visited, $depth + 1, $max_depth);
          $chain = array_merge($chain, $next_chain);
        }
      }
    }

    return $chain;
  }

  /**
   * Finds a redirect by path and query.
   *
   * @param string $path
   *   The path to search for.
   * @param array $query
   *   Query parameters.
   * @param string $language
   *   Language code.
   *
   * @return \Drupal\redirect\Entity\Redirect|null
   *   The redirect entity or NULL.
   */
  protected function findRedirectByPath($path, $query, $language) {
    $hash = Redirect::generateHash($path, $query, $language);
    $ids = $this->getRedirectStorage()->getQuery()
      ->accessCheck(FALSE)
      ->condition('hash', $hash)
      ->range(0, 1)
      ->execute();

    if (!$ids) {
      return NULL;
    }

    /** @var \Drupal\redirect\Entity\Redirect|null $redirect */
    $redirect = $this->getRedirectStorage()->load(reset($ids));
    return $redirect;
  }

  /**
   * Gets the logger channel.
   *
   * @return \Drupal\Core\Logger\LoggerChannelInterface
   *   The logger channel.
   */
  protected function getLogger() {
    if (!$this->logger) {
      $this->logger = $this->loggerFactory->get('redirect_audit');
    }
    return $this->logger;
  }

  /**
   * {@inheritdoc}
   */
  public function markAsProcessed(Redirect $redirect) {
    $this->database->merge('redirect_audit_processed')
      ->key(['rid' => $redirect->id()])
      ->fields(['chain_processed' => $this->time->getRequestTime()])
      ->execute();
  }

}
