<?php

namespace Drupal\Tests\redirect_audit\Kernel\Service;

use Drupal\Tests\redirect_audit\Kernel\RedirectAuditKernelTestBase;

/**
 * Tests for RedirectAuditFixer service.
 *
 * Essential tests for automatic chain resolution.
 *
 * @coversDefaultClass \Drupal\redirect_audit\Service\RedirectAuditFixer
 *
 * @group redirect_audit
 * @codingStandardsIgnoreFile
 */
class RedirectAuditFixerTest extends RedirectAuditKernelTestBase {

  /**
   * Tests fixer resolves simple chain.
   */
  public function testFixesSimpleChain(): void {
    // Arrange: Create and save chain a -> b -> c.
    $redirects = $this->createRedirectChain(['a', 'b', 'c']);
    $data = $this->analyzer->analyzeRedirect($redirects[0]);

    // Skip if no chain detected.
    if (!$data) {
      $this->markTestSkipped('No chain detected by analyzer');
    }

    $audit_id = $this->storage->saveChain(
      $data['source_rid'],
      $data['target_rid'],
      $data['path']
    );

    // Act: Fix chain.
    $result = $this->fixer->fixChain($audit_id);

    // Assert: Chain fixed, audit record removed.
    $this->assertTrue($result, 'Chain should be fixed successfully');
    $this->assertNull($this->storage->getChainById($audit_id), 'Audit record should be removed');

    // Verify redirect was updated.
    $redirect = $this->redirectStorage->load($redirects[0]->id());
    $target_url = $redirect->getRedirectUrl();

    // The URL should point to the final destination.
    $this->assertNotNull($target_url, 'Redirect should have a target URL');
  }

  /**
   * Tests fixer resolves complex chain.
   */
  public function testFixesComplexChain(): void {
    // Arrange: Long chain a -> b -> c -> d.
    $redirects = $this->createRedirectChain(['a', 'b', 'c', 'd']);
    $data = $this->analyzer->analyzeRedirect($redirects[0]);

    // Skip if no chain detected.
    if (!$data) {
      $this->markTestSkipped('No chain detected by analyzer');
    }

    $audit_id = $this->storage->saveChain(
      $data['source_rid'],
      $data['target_rid'],
      $data['path']
    );

    // Act: Fix chain.
    $result = $this->fixer->fixChain($audit_id);

    // Assert: Chain fixed.
    $this->assertTrue($result, 'Complex chain should be fixed');

    // Verify redirect exists and was modified.
    $redirect = $this->redirectStorage->load($redirects[0]->id());
    $this->assertNotNull($redirect, 'Redirect should still exist');
    $this->assertNotNull($redirect->getRedirectUrl(), 'Redirect should have target URL');
  }

  /**
   * Tests fixer handles invalid data gracefully.
   */
  public function testHandlesInvalidData(): void {
    // Act: Try to fix non-existent chain.
    $result = $this->fixer->fixChain(99999);

    // Assert: Returns false, no exception.
    $this->assertFalse($result);
  }

}
