<?php

declare(strict_types=1);

namespace Drupal\redirect_widget\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\node\Entity\NodeType;

/**
 * Configure Redirect Widget settings for this site.
 */
final class RedirectWidgetSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'redirect_widget_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['redirect_widget.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('redirect_widget.settings');
    // General configuration.
     $form['general'] = [
      '#type' => 'details',
      '#title' => $this->t('General settings'),
      '#open' => TRUE,
    ];

    $form['general']['widget_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Widget title'),
      '#default_value' => $config->get('widget_title') ?? 'Redirect URL',
      '#description' => $this->t('The title displayed for the redirect widget.'),
      '#required' => TRUE,
    ];

    $form['general']['widget_replace'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Replace default widget'),
      '#default_value' => $config->get('widget_replace') ?? TRUE,
      '#description' => $this->t('Replace the default redirect widget with this custom widget.'),
    ];

    $form['general']['widget_open'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Open widget by default'),
      '#default_value' => $config->get('widget_open') ?? FALSE,
      '#description' => $this->t('If checked, the redirect widget will be expanded by default when editing content.'),
    ];

    // Autofill configuration.
    $form['autofill'] = [
      '#type' => 'details',
      '#title' => $this->t('Autofill'),
      '#open' => TRUE,
    ];

    // Add "Enable autofill" checkbox.
    $form['autofill']['autofill_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable autofill'),
      '#default_value' => $config->get('autofill_enabled') ?? TRUE,
      '#description' => $this->t('Check to enable the autofill functionality for the redirect widget.'),
    ];

    // Load all content types for the checkboxes.
    $types = NodeType::loadMultiple();
    $options = [];
    foreach ($types as $type) {
      $options[$type->id()] = $type->label();
    }

    // Add the checkboxes for selecting which types to autofill.
    $form['autofill']['autofill_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Autofill content types'),
      '#options' => $options,
      '#default_value' => $config->get('autofill_types') ?: array_keys($options),
      '#description' => $this->t('Select which content types should be included when autofill is enabled.'),
      '#states' => [
        'visible' => [
          ':input[name="autofill_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('redirect_widget.settings')
        // General
      ->set('widget_title', $form_state->getValue('widget_title'))
      ->set('widget_replace', (bool) $form_state->getValue('widget_replace'))
      ->set('widget_open', (bool) $form_state->getValue('widget_open'))
      // Autofill
      ->set('autofill_enabled', (bool) $form_state->getValue('autofill_enabled'))
      ->set('autofill_types', array_filter($form_state->getValue('autofill_types')))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
