<?php

declare(strict_types=1);

namespace Drupal\redirect_widget;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\content_moderation\ModerationInformationInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\link\Plugin\Field\FieldWidget\LinkWidget;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\node\NodeInterface;
use Drupal\redirect\Entity\Redirect;
use Drupal\workflows\StateInterface;
use Drupal\workflows\WorkflowInterface;

/**
 * @todo Add class description.
 */
final class RedirectWidgetManager implements RedirectWidgetManagerInterface {

  use StringTranslationTrait;

  /**
   * Constructs a RedirectWidgetManager object.
   */
  public function __construct(
    private readonly ModuleHandlerInterface $moduleHandler,
    private readonly ModerationInformationInterface $moderationInformation,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly MessengerInterface $messenger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function injectRedirectWidget(array &$form, NodeInterface $node): void {
    $widgetConfig = $this->configFactory->get('redirect_widget.settings');

    $widgetTitle = $widgetConfig->get('widget_title') ?? 'Redirect URL';
    $widgetKey = (bool) $widgetConfig->get('widget_replace') ? 'url_redirects' : 'url_redirects_additional';
    
    $form[$widgetKey] = [
        '#type' => 'details',
        '#title' => $this->t($widgetTitle),
        '#tree' => TRUE,
        '#group' => 'advanced',
        '#open' => (bool) $widgetConfig->get('widget_open') ?? FALSE,
    ];
    $form[$widgetKey]['form'] = $this->getRedirectWidget($node);
    $form['actions']['submit']['#submit'][] = 'redirect_widget_submit_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getRedirectWidget(NodeInterface $node): array {
    $form = [];

    $defaultDestination = NULL;
    $defaultStatusCode = $this->configFactory->get('redirect.settings')->get('default_status_code');
    $processDefaultValue = FALSE;

    $existingRedirect = $this->getRedirectEntity($node);

    if ($existingRedirect instanceof Redirect) {
        $destinationUrl = $existingRedirect->getRedirectUrl();
        $defaultStatusCode = $existingRedirect->getStatusCode();

        // If the redirect is to an internal Node, we need to load that node as an entity to preprocess the field.
        if (!$destinationUrl->isExternal() && $destinationUrl->isRouted()) {
            $routeParameters = $destinationUrl->getRouteParameters();
            if (isset($routeParameters['node'])) {
                $defaultDestination = $routeParameters['node'] instanceof NodeInterface ? $routeParameters['node'] : Node::load($routeParameters['node']);
                $processDefaultValue = TRUE;
            }
        }

        // If no default destination was set in the previous step, we can just show the URL.
        if (empty($defaultDestination)) {
            $defaultDestination = $destinationUrl->toString();
        }

        // If the current redirect is unpublished, we should show a disclaimer
        if (!$existingRedirect->isPublished()) {
            $disclaimer = $this->t('This redirect is currently not published, it will be published when this page is published.');
        }
    }
    if (!empty($disclaimer)) {
        $form['#prefix'] = '<i>' . $disclaimer .'</i>';
    }

    $form['destination'] = [
        '#title' => $this->t('Redirect Destination'),
        '#maxlength' => 2048,
        '#required' => FALSE,
        '#element_validate' => [[LinkWidget::class, 'validateUriElement']],
        '#default_value' => $defaultDestination,
        '#process_default_value' => $processDefaultValue,

    ];

    $widgetConfig = $this->configFactory->get('redirect_widget.settings');

    // Check the field type we need to load based on the widget config.
    if((bool) $widgetConfig->get('autofill_enabled')) {
        $autofillTypes = $this->getAutofillTypes();
        $form['destination'] += [
            '#type' => 'entity_autocomplete',
            '#target_type' => 'node',
            '#selection_handler' => 'default:node',
            '#selection_settings' => [
                'target_bundles' => $autofillTypes,
            ],
            // Allow free text (URLs) to pass your LinkWidget validation.
            // This prevents entity reference validation from rejecting URLs.
            '#validate_reference' => FALSE,
            '#attributes' => [
                // Don’t trigger autocomplete when typing a raw path or anchor.
                'data-autocomplete-first-character-denylist' => '/#?',
            ],
        ];
    } 
    else {
        $form['destination'] += [
            '#type' => 'textfield',
        ];
    }

    $form['status_code'] = [
        '#type' => 'select',
        '#title' => $this->t('Redirect status'),
        '#description' => $this->t('You can find more information about HTTP redirect status codes at <a href="@status-codes">@status-codes</a>.', ['@status-codes' => 'http://en.wikipedia.org/wiki/List_of_HTTP_status_codes#3xx_Redirection']),
        '#default_value' => $defaultStatusCode,
        '#options' => $this->moduleHandler->invoke('redirect', 'status_code_options'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getRedirectEntity(NodeInterface $node): ?Redirect {
    $redirectManager = $this->entityTypeManager->getStorage('redirect');
    $query = $redirectManager->getQuery();
    $ids = $query
        ->condition('redirect_source.path', 'node/' . $node->id(), 'LIKE')
        ->accessCheck()
        ->execute();

    if (empty($ids)) return NULL;

    /** @var Redirect[] $redirects */
    $redirects = $redirectManager->loadMultiple($ids);
    if (empty($redirects)) return NULL;

    // If the node is translatable, the redirect should be loaded based on node language.
    if ($node->isTranslatable()) {
        $language = $node->language();
        foreach ($redirects as $redirect) {
            if ($redirect->language() === $language) return $redirect;
        }
        return NULL;
    }

    return reset($redirects);
  }

  /**
   * {@inheritdoc}
   */
  public function submitRedirectWidgetForm(FormStateInterface &$formState): void {
    $redirectValues = $formState->getValue(['url_redirects', 'form']);
    $node = $formState->getFormObject()->getEntity();
    if (!$node instanceof NodeInterface) return;

    $redirectEntity = $this->getRedirectEntity($node);

    // If no Redirect could be loaded, create a new one from this page.
    if (!$redirectEntity instanceof Redirect) {
        $redirectEntity = Redirect::create();
        $redirectEntity->setSource('node/' . $node->id());
    }

    // If there is no value set for this destination field, remove existing entities.
    if (empty($redirectValues['destination'])) {

        // if the page is still in an unpublished state, don't allow the redirect to be removed from the UI.
        if (!$this->isModerationStatePublished($formState) && $redirectEntity->isPublished()) return;
        $destination = $redirectEntity->getRedirectUrl();
        if (!empty($destination)) {
            $this->messenger->addStatus(
              $this->t('Redirect to "@destination" is removed.', ['@destination' => $destination->toString()])
            );
        }
        $redirectEntity->delete();
        return;
    }

    $destination = Url::fromUri($redirectValues['destination'])->toString();
    $redirectEntity->setRedirect($destination);
    $redirectEntity->setStatusCode($redirectValues['status_code']);
    if ($node->isTranslatable()) {
        $redirectEntity->setLanguage($node->language()->getId());
    }


    /**
     * If the moderation state is not published and the redirect has not been published before,
     * we should unpublish the redirect. Else, the redirect should be published.
     **/
    if (
        !$this->isModerationStatePublished($formState) &&
        ($redirectEntity->isNew() || !$redirectEntity->isPublished())
    ) {
        $redirectEntity->setUnpublished();
    } else {
        $redirectEntity->setPublished();
        if ($redirectEntity->isNew()) {
            $this->messenger->addStatus(
                $this->t('"@title" is now redirected to "@destination".', ['@title' => $node->getTitle(), "@destination" => $redirectEntity->getRedirectUrl()->toString()]),
            );
        }
    }
    $redirectEntity->save();
  }

  /**
   * Check if the form submission was for a published moderation state.
   *
   * @param FormStateInterface $formState
   *   The form state.
   *
   * @return bool
   *   TRUE = published state, FALSE = unpublished state.
   */
  protected function isModerationStatePublished(FormStateInterface $formState): bool {
    if(!$this->moduleHandler->moduleExists('content_moderation')) return TRUE;

    
    $node = $formState->getFormObject()->getEntity();

    if ($formState->hasValue('moderation_state')) {
        $moderationState = $formState->getValue('moderation_state');
        $moderationState = reset($moderationState)['value'];
        $workflow = $this->moderationInformation->getWorkflowForEntity($node);

        if (!$workflow instanceof WorkflowInterface) return TRUE;

        $actual_state = $workflow->getTypePlugin()->getState($moderationState);
        return $actual_state instanceof StateInterface ? $actual_state->isPublishedState() : TRUE;
    }
    return TRUE;
  }

  protected function getAutofillTypes(): array {
    $autofillTypes = $this->configFactory->get('redirect_widget.settings')->get('autofill_types') ?? [];
    $existingTypes = array_keys(NodeType::loadMultiple());
    $autofillTypes = array_values(array_intersect($autofillTypes, $existingTypes));
    return $autofillTypes;
  }

}
