<?php

namespace Drupal\redirect_entity_manager\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Configure Redirect Entity Manager settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a SettingsForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['redirect_entity_manager.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'redirect_entity_manager_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('redirect_entity_manager.settings');

    $form['enabled_entity_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Enabled entity types'),
      '#description' => $this->t('Select which entity types should display the redirect management tab.'),
      '#options' => $this->getContentEntityTypes(),
      '#default_value' => $config->get('enabled_entity_types') ?: ['node'],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $enabled_types = array_filter($form_state->getValue('enabled_entity_types'));
    $this->config('redirect_entity_manager.settings')
      ->set('enabled_entity_types', array_values($enabled_types))
      ->save();

    parent::submitForm($form, $form_state);
    // Clear routing cache to rebuild local tasks.
    \Drupal::service('router.builder')->rebuild();
  }

  /**
   * Get supported entity types.
   *
   * @return array
   *   An array of entity type labels keyed by entity type ID.
   */
  protected function getContentEntityTypes() {
    $supported_types = ['node', 'taxonomy_term', 'media'];
    $entity_types = [];
    foreach ($supported_types as $type_id) {
      $definition = $this->entityTypeManager->getDefinition($type_id, FALSE);
      if ($definition) {
        $entity_types[$type_id] = $definition->getLabel();
      }
    }

    return $entity_types;
  }

}
