<?php

namespace Drupal\redirect_entity_manager\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Url;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Controller for managing entity redirects.
 */
class RedirectEntityManagerController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a RedirectEntityManagerController object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(Connection $database, ConfigFactoryInterface $config_factory) {
    $this->database = $database;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('config.factory')
    );
  }

  /**
   * Checks access for the redirect management page.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   * @param string $entity_type
   *   The entity type ID.
   * @param mixed $node
   *   The node entity (if applicable).
   * @param mixed $taxonomy_term
   *   The taxonomy term entity (if applicable).
   * @param mixed $media
   *   The media entity (if applicable).
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(AccountInterface $account, $entity_type = NULL, $node = NULL, $taxonomy_term = NULL, $media = NULL) {
    $config = $this->configFactory->get('redirect_entity_manager.settings');
    $enabled_types = $config->get('enabled_entity_types') ?: ['node'];

    // Determine entity type from parameters if not provided.
    if (!$entity_type) {
      if ($node) {
        $entity_type = 'node';
      }
      elseif ($taxonomy_term) {
        $entity_type = 'taxonomy_term';
      }
      elseif ($media) {
        $entity_type = 'media';
      }
    }

    if (!in_array($entity_type, $enabled_types)) {
      return AccessResult::forbidden();
    }

    return AccessResult::allowedIfHasPermission($account, 'administer redirects');
  }

  /**
   * Displays and manages redirects for the current entity.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param mixed $node
   *   The node entity (if applicable).
   * @param mixed $taxonomy_term
   *   The taxonomy term entity (if applicable).
   * @param mixed $media
   *   The media entity (if applicable).
   *
   * @return array
   *   A render array.
   */
  public function manageRedirects($entity_type = NULL, $node = NULL, $taxonomy_term = NULL, $media = NULL) {
    // Get the entity from the appropriate parameter.
    $entity = $node ?: $taxonomy_term ?: $media;

    if (!$entity) {
      return ['#markup' => $this->t('Entity not found.')];
    }

    // Determine entity type from the entity if not provided.
    if (!$entity_type) {
      $entity_type = $entity->getEntityTypeId();
    }
    $rows = [];

    // Get the internal path for this entity.
    $internal_path = '/' . $entity_type . '/' . $entity->id();

    // Get path alias if exists.
    $alias_manager = \Drupal::service('path_alias.manager');
    $alias = $alias_manager->getAliasByPath($internal_path);

    // Query the redirect table for redirects pointing to OR from this entity.
    $query = $this->database->select('redirect', 'r')
      ->fields('r', ['rid', 'redirect_source__path', 'redirect_source__query', 'redirect_redirect__uri', 'status_code', 'language']);

    // Add condition group for redirects TO or FROM this entity (including alias).
    $or_group = $query->orConditionGroup()
      ->condition('redirect_redirect__uri', 'internal:' . $internal_path)
      ->condition('redirect_source__path', ltrim($internal_path, '/'));

    // Also check for alias if it's different from the system path.
    if ($alias !== $internal_path) {
      $or_group->condition('redirect_redirect__uri', 'internal:' . $alias);
      $or_group->condition('redirect_source__path', ltrim($alias, '/'));
    }

    $query->condition($or_group);

    $results = $query->execute()->fetchAll();

    // Get language manager for language names.
    $language_manager = \Drupal::languageManager();

    // Separate and sort results.
    $from_redirects = [];
    $to_redirects = [];

    foreach ($results as $result) {
      // Determine if this is a "from" redirect.
      $is_from = (ltrim($internal_path, '/') === $result->redirect_source__path ||
                  ($alias !== $internal_path && ltrim($alias, '/') === $result->redirect_source__path));

      if ($is_from) {
        $from_redirects[] = $result;
      }
      else {
        $to_redirects[] = $result;
      }
    }

    // Sort by rid descending (newest first).
    usort($from_redirects, function ($a, $b) {
      return $b->rid - $a->rid;
    });
    usort($to_redirects, function ($a, $b) {
      return $b->rid - $a->rid;
    });

    // Merge: from redirects first, then to redirects.
    $results = array_merge($from_redirects, $to_redirects);

    $rows = [];
    $add_separator = !empty($from_redirects) && !empty($to_redirects);
    $from_count = count($from_redirects);
    $current_index = 0;

    // Add "Redirects FROM this content paths" header if there are from redirects.
    if (!empty($from_redirects)) {
      $rows[] = [
        [
          'data' => [
            '#markup' => '<strong>' . $this->t('Redirects FROM this content paths') . '</strong>',
          ],
          'colspan' => 5,
          'class' => ['redirect-section-header'],
          'style' => 'text-align: center;',
        ],
      ];
    }

    foreach ($results as $result) {
      $current_index++;
      $source_path = $result->redirect_source__path;
      if (!empty($result->redirect_source__query)) {
        $query_params = unserialize($result->redirect_source__query);
        if (is_array($query_params)) {
          $source_path .= '?' . http_build_query($query_params);
        }
      }

      // Parse the destination URI.
      $destination_uri = $result->redirect_redirect__uri;
      $destination_path = str_replace(['internal:', 'entity:'], '', $destination_uri);

      // Get language name.
      $language_name = $this->t('All');
      if (!empty($result->language)) {
        $language = $language_manager->getLanguage($result->language);
        if ($language) {
          $language_name = $language->getName();
        }
      }

      $rows[] = [
        [
          'data' => [
            '#type' => 'link',
            '#title' => '/' . $source_path,
            '#url' => Url::fromUserInput('/' . $result->redirect_source__path),
          ],
        ],
        $destination_path,
        $result->status_code,
        $language_name,
        [
          'data' => [
            '#type' => 'operations',
            '#links' => [
              'edit' => [
                'title' => $this->t('Edit'),
                'url' => Url::fromRoute('entity.redirect.edit_form', ['redirect' => $result->rid], [
                  'query' => [
                    'destination' => Url::fromRoute('redirect_entity_manager.redirects_' . $entity_type, [$entity_type => $entity->id()])->toString(),
                  ],
                ]),
              ],
              'delete' => [
                'title' => $this->t('Delete'),
                'url' => Url::fromRoute('entity.redirect.delete_form', ['redirect' => $result->rid], [
                  'query' => [
                    'destination' => Url::fromRoute('redirect_entity_manager.redirects_' . $entity_type, [$entity_type => $entity->id()])->toString(),
                  ],
                ]),
              ],
              'test' => [
                'title' => $this->t('Test'),
                'url' => Url::fromUserInput('/' . $result->redirect_source__path),
                'attributes' => ['target' => '_blank'],
              ],
            ],
          ],
        ],
      ];

      // Add "Redirects TO this content paths" header after from redirects.
      if ($add_separator && $current_index === $from_count) {
        $rows[] = [
          [
            'data' => [
              '#markup' => '<strong>' . $this->t('Redirects TO this content paths') . '</strong>',
            ],
            'colspan' => 5,
            'class' => ['redirect-section-header'],
            'style' => 'text-align: center;',
          ],
        ];
      }
    }

    $build = [];

    // Add action links for creating redirects.
    $build['actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['action-links']],
    ];

    // Determine the route name based on entity type.
    $add_redirect_route = 'redirect_entity_manager.add_redirect_form_' . $entity_type;

    $build['actions']['add_redirect_from'] = [
      '#type' => 'link',
      '#title' => $this->t('Redirect from this content'),
      '#url' => Url::fromRoute($add_redirect_route, [$entity_type => $entity->id()]),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
      ],
    ];

    $build['actions']['add_redirect_to'] = [
      '#type' => 'link',
      '#title' => $this->t('Redirect to this content'),
      '#url' => Url::fromRoute('redirect.add', [], [
        'query' => [
          'redirect' => ltrim($internal_path, '/'),
          'destination' => Url::fromRoute('redirect_entity_manager.redirects_' . $entity_type, [$entity_type => $entity->id()])->toString(),
        ],
      ]),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
      ],
    ];

    $build['redirects_table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('From'),
        $this->t('To'),
        $this->t('Status Code'),
        $this->t('Language'),
        $this->t('Operations'),
      ],
      '#rows' => $rows,
      '#empty' => $this->t('No redirects found pointing to this content.'),
    ];

    return $build;
  }

}
