<?php

namespace Drupal\redirect_entity_manager\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\redirect\Entity\Redirect;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\redirect\RedirectRepository;

/**
 * Form for adding redirects from an entity.
 */
class AddRedirectForm extends FormBase {

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The path alias manager.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected $aliasManager;

  /**
   * The redirect repository.
   *
   * @var \Drupal\redirect\RedirectRepository
   */
  protected $redirectRepository;

  /**
   * Constructs an AddRedirectForm object.
   *
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\path_alias\AliasManagerInterface $alias_manager
   *   The path alias manager.
   * @param \Drupal\redirect\RedirectRepository $redirect_repository
   *   The redirect repository.
   */
  public function __construct(LanguageManagerInterface $language_manager, AliasManagerInterface $alias_manager, RedirectRepository $redirect_repository) {
    $this->languageManager = $language_manager;
    $this->aliasManager = $alias_manager;
    $this->redirectRepository = $redirect_repository;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('language_manager'),
      $container->get('path_alias.manager'),
      $container->get('redirect.repository')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'redirect_entity_manager_add_redirect_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $entity_type = NULL, $node = NULL, $taxonomy_term = NULL, $media = NULL) {
    // Get the entity from the appropriate parameter.
    $entity = $node ?: $taxonomy_term ?: $media;
    if (!$entity) {
      $form['error'] = [
        '#markup' => $this->t('Entity not found.'),
      ];
      return $form;
    }
    // Determine entity type from the entity if not provided.
    if (!$entity_type) {
      $entity_type = $entity->getEntityTypeId();
    }
    // Store entity info in form state.
    $form_state->set('entity', $entity);
    $form_state->set('entity_type', $entity_type);
    $internal_path = '/' . $entity_type . '/' . $entity->id();
    $alias = $this->aliasManager->getAliasByPath($internal_path);

    $form['info'] = [
      '#type' => 'item',
      '#markup' => $this->t('Create redirects from this content to another destination.'),
    ];

    $form['source_type'] = [
      '#type' => 'radios',
      '#title' => $this->t('Path to redirect from'),
      '#options' => [
        'node_path' => $this->t('System path: @path', ['@path' => $internal_path]),
        'alias_path' => $this->t('URL alias: @alias', ['@alias' => $alias !== $internal_path ? $alias : $this->t('(No alias)')]),
        'both' => $this->t('Both system path and URL alias'),
      ],
      '#default_value' => 'both',
      '#required' => TRUE,
    ];
    $form['redirect_to'] = [
      '#type' => 'textfield',
      '#title' => $this->t('To'),
      '#description' => $this->t('Start typing the title of a piece of content to select it. You can also enter an internal path such as %add-node or an external URL such as %url. Enter %front to link to the front page. Enter %nolink to display link text only. Enter %button to display keyboard-accessible link text only.', ['%front' => '<front>', '%add-node' => '/node/add', '%url' => 'https://example.com', '%nolink' => '<nolink>', '%button' => '<button>']),
      '#required' => TRUE,
    ];
    $form['status_code'] = [
      '#type' => 'select',
      '#title' => $this->t('Redirect status'),
      '#description' => $this->t('You can find more information about HTTP redirect status codes at <a href="http://en.wikipedia.org/wiki/List_of_HTTP_status_codes#3xx_Redirection" target="_blank">http://en.wikipedia.org/wiki/List_of_HTTP_status_codes#3xx_Redirection</a>.'),
      '#options' => [
        '301' => $this->t('301 Moved Permanently'),
        '302' => $this->t('302 Found'),
        '303' => $this->t('303 See Other'),
        '307' => $this->t('307 Temporary Redirect'),
      ],
      '#default_value' => '301',
      '#required' => TRUE,
    ];
    $form['language'] = [
      '#type' => 'language_select',
      '#title' => $this->t('Language'),
      '#description' => $this->t('The redirect language.'),
      '#default_value' => $entity->language()->getId(),
    ];
    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#button_type' => 'primary',
    ];
    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => Url::fromRoute('redirect_entity_manager.redirects_' . $entity_type, [$entity_type => $entity->id()]),
      '#attributes' => ['class' => ['button']],
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $entity = $form_state->get('entity');
    $entity_type = $form_state->get('entity_type');
    $source_type = $form_state->getValue('source_type');
    $redirect_to = $form_state->getValue('redirect_to');
    $status_code = $form_state->getValue('status_code');
    $language = $form_state->getValue('language');
    $internal_path = '/' . $entity_type . '/' . $entity->id();
    $alias = $this->aliasManager->getAliasByPath($internal_path);
    // Normalize destination.
    if (!preg_match('/^https?:\/\//', $redirect_to)) {
      $redirect_to = 'internal:/' . ltrim($redirect_to, '/');
    }
    $created_count = 0;
    $skipped_count = 0;

    // Create redirect from system path.
    if (in_array($source_type, ['node_path', 'both'])) {
      $source_path = ltrim($internal_path, '/');

      // Check if redirect already exists.
      $existing = $this->redirectRepository->findBySourcePath($source_path, $language);

      if (!$existing) {
        $redirect = Redirect::create([
          'redirect_source' => $source_path,
          'redirect_redirect' => $redirect_to,
          'language' => $language,
          'status_code' => $status_code,
        ]);
        $redirect->save();
        $created_count++;
      }
      else {
        $skipped_count++;
        $this->messenger()->addWarning($this->t('A redirect from %path already exists.', ['%path' => '/' . $source_path]));
      }
    }

    // Create redirect from alias path.
    if (in_array($source_type, ['alias_path', 'both']) && $alias !== $internal_path) {
      $source_path = ltrim($alias, '/');

      // Check if redirect already exists.
      $existing = $this->redirectRepository->findBySourcePath($source_path, $language);

      if (!$existing) {
        $redirect = Redirect::create([
          'redirect_source' => $source_path,
          'redirect_redirect' => $redirect_to,
          'language' => $language,
          'status_code' => $status_code,
        ]);
        $redirect->save();
        $created_count++;
      }
      else {
        $skipped_count++;
        $this->messenger()->addWarning($this->t('A redirect from %path already exists.', ['%path' => $alias]));
      }
    }

    if ($created_count > 0) {
      $this->messenger()->addStatus($this->t('Created @count redirect(s).', ['@count' => $created_count]));
    }
    // Redirect back to the redirect management page.
    $form_state->setRedirect('redirect_entity_manager.redirects_' . $entity_type, [$entity_type => $entity->id()]);
  }

}
