<?php

namespace Drupal\redis_cache_clear\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\redis\ClientFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Form\FormStateInterface;

class RedisCacheClearController extends ControllerBase {

  protected $redisFactory;

  // Prefix for filtering keys.
  protected $prefix = 'drupal.redis.9.5.11';

  public function __construct(ClientFactory $redisFactory) {
    $this->redisFactory = $redisFactory;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('redis.factory')
    );
  }

  /**
   * Display Redis keys and Clear button.
   */
  public function listKeys() {
    // Add Clear button form.
    $build['clear_form'] = [
      '#type' => 'form',
      '#form_id' => 'redis_cache_clear_form',
      'actions' => [
        'submit' => [
          '#type' => 'submit',
          '#value' => $this->t('Clear Redis Cache (prefix: @prefix)', ['@prefix' => $this->prefix]),
          '#button_type' => 'primary',
          '#submit' => [[$this, 'clearCacheSubmit']],
        ],
      ],
    ];

    return $build;
  }

  /**
   * Clear cache via form submit.
   */
  public function clearCacheSubmit(array &$form, FormStateInterface $form_state) {
    $this->clearRedisData();
    $this->messenger()->addStatus($this->t('Redis cache (prefix: @prefix) cleared successfully.', ['@prefix' => $this->prefix]));
    $form_state->setRedirect('redis_cache_clear.admin');
  }

  /**
   * Clear cache via direct route (for POST).
   */
  public function clearCache(Request $request) {
    $this->clearRedisData();
    $this->messenger()->addStatus($this->t('Redis cache (prefix: @prefix) cleared successfully.', ['@prefix' => $this->prefix]));
    return new RedirectResponse(Url::fromRoute('redis_cache_clear.admin')->toString());
  }

  /**
   * Reusable function to clear Redis keys with prefix.
   */
  protected function clearRedisData() {
    $client = \Drupal::service('redis.factory')->getClient();

    $iterator = null;
    while ($arr = $client->scan($iterator, $this->prefix . '*', 500)) {
      foreach ($arr as $key) {
        $client->del($key);
      }
    }
  }

}
