<?php

namespace Drupal\redis_cache_clear\Cron;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\State\StateInterface;

class ClearRedisCache {

  protected $logger;
  protected $state;

  // Prefix for filtering keys.
  protected $prefix;

  // Run only once every 24 hours.
  protected $interval; // define variable for set time in seconds

  public function __construct(LoggerChannelInterface $logger, StateInterface $state) {
    $this->logger = $logger;
    $this->state = $state;
    // Load interval from config.
    $config = \Drupal::config('redis_cache_clear.settings');
    $this->interval = (int) $config->get('interval') ?: 86400;
    // Get prefix from config or fall back to default.
    $manual_prefix = $config->get('prefix');
     // If no custom prefix provided, use Drupal's default Redis prefix.
    if (!empty($manual_prefix)) {
      $this->prefix = $manual_prefix;
    }
    else {
      // Get Drupal's default Redis prefix from the Redis service.
      /** @var \Drupal\redis\ClientFactory $redis_factory */
      $redis_factory = \Drupal::service('redis.factory');
      $client = $redis_factory->getClient();

      // The prefix is defined in the Redis settings, typically as 'cache_prefix'.
      $this->prefix = $client->getOption(\Redis::OPT_PREFIX) ?: '';
    }
  }

  /**
   * Cron job: Clears Redis keys with specific prefix once per day.
   */
  public function clear() {
    $last_run = $this->state->get('redis_cache_clear.last_run', 0);
    $now = \Drupal::time()->getCurrentTime();
    if (($now - $last_run) < $this->interval) {
      // Not yet time to run.
      return;
    }

    $client = \Drupal::service('redis.factory')->getClient();
    $deleted = 0;
    $iterator = null;

    while ($arr = $client->scan($iterator, $this->prefix . '*', 500)) {
      foreach ($arr as $key) {
        $client->del($key);
        $deleted++;
      }
    }

    // Save last run timestamp.
    $this->state->set('redis_cache_clear.last_run', $now);

    $this->logger->notice('Cleared @count Redis keys with prefix "@prefix".', [
      '@count' => $deleted,
      '@prefix' => $this->prefix ?: '[default Drupal prefix]',
    ]);
  }
}
