<?php

namespace Drupal\redis_cache_clear\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\redis\ClientFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Form\FormStateInterface;

class RedisCacheClearController extends ControllerBase {

  protected $redisFactory;

  public function __construct(ClientFactory $redisFactory) {
    $this->redisFactory = $redisFactory;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('redis.factory')
    );
  }

  /**
  * Resolve Redis prefix from database name.
  */
  protected function prefix(): string {
    $connection = \Drupal::database();
    $options = $connection->getConnectionOptions();
 
    $database = $options['database'] ?? 'default';
 
    return strtolower(preg_replace('/[^a-zA-Z0-9_]/', '_', $database)) . '_';
  }
 
  /**
   * Display only Clear button.
   */
  public function listKeys() {
    $prefix = $this->prefix();

    $build['clear_form'] = [
      '#type' => 'form',
      '#form_id' => 'redis_cache_clear_form',
      'actions' => [
        'submit' => [
          '#type' => 'submit',
          '#value' => $this->t(
            'Clear Redis Cache (prefix: @prefix)',
            ['@prefix' => $prefix]
          ),
          '#button_type' => 'primary',
          '#submit' => [[$this, 'clearCacheSubmit']],
        ],
      ],
    ];

    return $build;
  }

  /**
   * Clear cache via form submit.
   */
  public function clearCacheSubmit(array &$form, FormStateInterface $form_state) {
    $deleted = $this->clearRedisData();
    $prefix = $this->prefix();

    $this->messenger()->addStatus($this->t(
      'Cleared @count Redis keys with prefix "@prefix".',
      ['@count' => $deleted, '@prefix' => $prefix]
    ));
 
    $form_state->setRedirect('redis_cache_clear.admin');
  }

  /**
   * Clear cache via direct route (POST).
   */
  public function clearCache(Request $request) {
    $deleted = $this->clearRedisData();
    $prefix = $this->prefix();
 
    $this->messenger()->addStatus($this->t(
      'Cleared @count Redis keys with prefix "@prefix".',
      ['@count' => $deleted, '@prefix' => $prefix]
    ));

    return new RedirectResponse(
      Url::fromRoute('redis_cache_clear.admin')->toString()
    );
  }

  /**
   * Reusable function to clear Redis keys with prefix.
   */
  protected function clearRedisData(): int {
    $client = \Drupal::service('redis.factory')->getClient();
    $prefix = $this->prefix();
 
    $iterator = 0;
    $deleted = 0;
 
    while ($arr = $client->scan($iterator, $prefix . '*', 500)) {
      foreach ($arr as $key) {
        $client->del((string) $key);
        $deleted++;
      }
    }

    \Drupal::logger('redis_cache_clear')->notice(
      'Deleted @count Redis keys with prefix "@prefix".',
      ['@count' => $deleted, '@prefix' => $prefix]
    );
 
    return $deleted;
  }

}