<?php

namespace Drupal\redis_cache_clear\Cron;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\State\StateInterface;

class ClearRedisCache {

  protected $logger;
  protected $state;
 
  // Run only once every 24 hours.
  protected $interval = 86400; // 1 day in seconds

  public function __construct(LoggerChannelInterface $logger, StateInterface $state) {
    $this->logger = $logger;
    $this->state = $state;
  }

  /*
    Resolve Redis prefix from database name.
   */
  protected function prefix(): string {
    $connection = \Drupal::database();
    $options = $connection->getConnectionOptions();
 
    $database = $options['database'] ?? 'default';
 
    return strtolower(preg_replace('/[^a-zA-Z0-9_]/', '_', $database)) . '_';
  }

  /*
    Cron job: Clears Redis keys with database-based prefix once per day.
   */
  public function clear() {
    $last_run = $this->state->get('redis_cache_clear.last_run', 0);
    $now = \Drupal::time()->getCurrentTime();

    // Enforce interval.
    if (($now - $last_run) < $this->interval) {
      return;
    }

    $client = \Drupal::service('redis.factory')->getClient();
    $prefix = $this->prefix();

    $deleted = 0;
    $iterator = 0;

    while ($keys = $client->scan($iterator, $prefix . '', 500)) {
      foreach ($keys as $key) {
        $client->del((string) $key);
        $deleted++;
      }
    }

    // Store last successful run time.
    $this->state->set('redis_cache_clear.last_run', $now);
 
    $this->logger->notice(
      'Cron cleared @count Redis keys with prefix "@prefix".',
      [
        '@count' => $deleted,
        '@prefix' => $prefix,
      ]
    );
  }

}