<?php

namespace Drupal\reference_by_view_mode\Plugin\EntityReferenceSelection;

use Drupal\Component\Plugin\Definition\PluginDefinitionInterface;
use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\TranslatableInterface;
use Drupal\Core\Url;
use Drupal\Core\StringTranslation\StringTranslationTrait;

trait ViewModeSelectionTrait {

  use StringTranslationTrait;

  public function defaultConfiguration(): array {
    return [
        // When target_view_modes is empty, do not filter.
      'target_view_modes' => NULL,
      'multiple_operator' => 'OR',
      'sort' => [
        'field' => '_none',
        'direction' => 'ASC',
      ],
    ] + parent::defaultConfiguration();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);
    unset($form['negate'], $form['target_bundles'], $form['target_bundles_drag_drop']);

    $configuration = $this->getConfiguration();
    $multiple_operator = $configuration['multiple_operator'] ?? 'OR';
    assert(is_string($multiple_operator));
    $form['multiple_operator'] = $this->switchMultipleOperator($multiple_operator);

    $entity_type_id = $configuration['target_type'];
    assert(is_string($entity_type_id));
    $view_modes = (array) ($this->getConfiguration()['target_view_modes'] ?? []);

    $form['target_view_modes'] = $this->checkboxesViewModes($entity_type_id, $view_modes);

    $bundles = $this->viewModesToBundles($entity_type_id, $view_modes, $multiple_operator);
    $form['current_bundles'] = [
      '#type' => 'markup',
      '#markup' => $this->t(
        '<p>Currently targeting bundles: %bundles</p>',
        ['%bundles' => !empty($bundles) ? implode(', ', $bundles) : $this->t('- None -')]
      ),
    ];
    $form['administer_link'] = [
      '#type' => 'link',
      '#url' => Url::fromRoute('entity.entity_view_mode.collection'),
      '#title' => $this->t('Manage view modes.'),
    ];

    assert(is_string($configuration['sort']['field']) || is_null($configuration['sort']['field']));
    assert(is_string($configuration['sort']['direction']) || is_null($configuration['sort']['direction']));
    $form['sort'] = $this->configureSort($entity_type_id, $configuration['sort']['field'] ?? '_none', $configuration['sort']['direction'] ?? 'ASC');

    return $form;
  }

  protected function switchMultipleOperator(string $current_value): array {
    $options = [
      'OR' => $this->t('or'),
      'AND' => $this->t('and'),
    ];

    return [
      '#type' => 'radios',
      '#title' => $this->t('Operator'),
      '#options' => $options,
      '#default_value' => $current_value,
      '#required' => TRUE,
      '#size' => 6,
      '#multiple' => FALSE,
      '#ajax' => TRUE,
    ];
  }

  protected function checkboxesViewModes(string $entity_type_id, array $current_values): array {
    $view_modes = $this->entityDisplayRepository->getViewModes($entity_type_id);
    $view_modes = array_combine(array_keys($view_modes), array_column($view_modes, 'label'));

    if (empty($view_modes)) {
      return ['#type' => 'value', '#value' => $current_values];
    }

    return [
      '#type' => 'checkboxes',
      '#title' => $this->t('View modes'),
      '#options' => $view_modes,
      '#default_value' => $current_values,
      '#required' => FALSE,
      '#size' => 6,
      '#multiple' => TRUE,
      '#ajax' => TRUE,
      '#element_validate' => [
        [
          static::class,
          'settingsFormValidate',
        ],
      ],
    ];
  }

  protected function configureSort(string $entity_type_id, string $field_value, string $direction_value): array {
    $fields = [];

    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
    if ($id_key = $entity_type->getKey('id')) {
      $fields[$id_key] = $this->t('Id');
    }
    if ($label_key = $entity_type->getKey('label')) {
      $fields[$label_key] = $this->t('Label');
    }
    if (empty($fields)) {
      return [
        'field' => [
          '#type' => 'value',
          '#value' => '_none',
        ],
      ];
    }

    $fields = ['_none' => $this->t('- None -')] + $fields;

    return [
      'field' => [
        '#type' => 'radios',
        '#title' => $this->t('Sort by'),
        '#options' => $fields,
        '#ajax' => TRUE,
        '#empty_value' => '_none',
        '#default_value' => $field_value,
      ],
      'direction' => [
        '#type' => 'radios',
        '#title' => $this->t('Direction'),
        '#options' => [
          'ASC' => $this->t('Ascending'),
          'DESC' => $this->t('Descending'),
        ],
        '#default_value' => strtoupper($direction_value),
        '#required' => FALSE,
        '#size' => 6,
        '#multiple' => TRUE,
        '#ajax' => TRUE,
      ],
    ];
  }

  /**
   * @return array<string, string>
   */
  protected function viewModesToBundles(string $entity_type_id, array $view_modes, string $multiple_operator): array {
    $bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);
    $result = [];

    foreach ($bundles as $bundle => $name_info) {
      assert(is_string($bundle));
      $view_modes_form_bundle = $this->entityDisplayRepository->getViewModeOptionsByBundle($entity_type_id, $bundle);
      $matching_view_modes = array_intersect_key($view_modes, $view_modes_form_bundle);

      if (match ($multiple_operator) {
        'OR' => !empty($matching_view_modes) || empty($view_modes),
        'AND' => count($view_modes) === count($matching_view_modes) && !empty($view_modes),
        default => FALSE
      }) {
      assert(is_string($name_info['label']));
      $result[$bundle] = $name_info['label'];
      }
    }

    return $result;
  }

  public function getReferenceableEntities($match = NULL, $match_operator = 'CONTAINS', $limit = 0): array {
    $target_type = $this->getConfiguration()['target_type'];
    assert(is_string($target_type));

    /** @var array<int|string>|int $result */
    $result = $this->buildEntityQuery($match, $match_operator)
      ->range($limit > 0 ? 0 : NULL, $limit ?: NULL)
      ->execute();

    if (empty($result) || !is_array($result)) {
      return [];
    }

    $options = [];
    $entities = $this->entityTypeManager->getStorage($target_type)->loadMultiple($result);
    $current_language = \Drupal::languageManager()->getCurrentLanguage(LanguageInterface::TYPE_CONTENT)->getId();
    foreach ($entities as $entity_id => $entity) {
      $bundle = $entity->bundle();

      if ($entity instanceof TranslatableInterface && $entity->hasTranslation($current_language)) {
        $entity = $entity->getTranslation($current_language);
      }

      $options[$bundle][$entity_id] = Html::escape($entity->label() ?? "{$target_type}:{$bundle}:{$entity_id}");
    }

    return $options;
  }

  public function countReferenceableEntities($match = NULL, $match_operator = 'CONTAINS'): int {
    /** @var int */
    return $this->buildEntityQuery($match, $match_operator)->count()->execute();
  }

  public function validateReferenceableEntities(array $ids): array {
    if (!empty($ids)) {
      $entity_type = $this->entityTypeManager->getDefinition($this->configuration['target_type']);
      $query = $this->buildEntityQuery()
        ->condition($entity_type->getKey('id') ?: 'id', $ids, 'IN');
      /** @var string[]|int[] */
      return $query->execute();
    }

    return [];
  }

  public function calculateDependencies(): array {
    $dependencies = parent::calculateDependencies();
    $configuration = $this->getConfiguration();
    $entity_type_id = $configuration['target_type'];
    assert(is_string($entity_type_id));

    if (!empty($configuration['target_view_modes'])
      && is_array($configuration['target_view_modes'])) {
      /** @var array<int|string, string> $target_view_modes */
      $target_view_modes = $configuration['target_view_modes'];
      array_walk($target_view_modes, fn(string|int &$view_mode_id): string => $view_mode_id = "{$entity_type_id}.{$view_mode_id}");

      foreach ($this->entityTypeManager->getStorage('entity_view_mode')->loadMultiple($target_view_modes) as $view_mode) {
        $dependencies[$view_mode->getConfigDependencyKey()][] = $view_mode->getConfigDependencyName();
      }
    }

    $plugin_definition = $this->getPluginDefinition();
    if ($plugin_definition instanceof PluginDefinitionInterface) {
      $dependencies['module'][] = $plugin_definition->getProvider();
    }
    elseif (isset($plugin_definition['provider'])) {
      $dependencies['module'][] = $plugin_definition['provider'];
    }

    return $dependencies;
  }

  public static function settingsFormValidate(array $element, FormStateInterface $form_state, array $form): void {
    if ($form_state->getValue([
      'settings',
      'handler_settings',
      'multiple_operator',
    ]) === 'AND') {
      if (empty($element['#value'])) {
        $form_state->setError($element, new TranslatableMarkup('You have to select at least one view mode.'));
      }
    }

    assert(is_array($element['#value']));
    $element['#value'] = array_filter($element['#value']);
    $form_state->setValueForElement($element, $element['#value']);
  }

  /**
   * @param string|null $match
   * @param string $match_operator
   */
  protected function buildEntityQuery($match = NULL, $match_operator = 'CONTAINS'): QueryInterface {
    $configuration = $this->getConfiguration();
    $entity_type_id = $configuration['target_type'];
    assert(is_string($entity_type_id));
    $view_modes = (array) ($configuration['target_view_modes'] ?? []);
    $multiple_operator = $configuration['multiple_operator'] ?? 'OR';
    assert(is_string($multiple_operator));

    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);

    $bundles = $this->viewModesToBundles($entity_type_id, $view_modes, $multiple_operator);

    $query = $this->entityTypeManager->getStorage($entity_type_id)->getQuery();
    $query->accessCheck(TRUE);
    $query->addTag($entity_type_id . '_access');
    $query->addTag('entity_reference');
    $query->addMetaData('entity_reference_selection_handler', $this);
    $query->condition($entity_type->getKey('bundle') ?: 'type', array_keys($bundles), 'IN');

    if (isset($match) && $label_key = $entity_type->getKey('label')) {
      $query->condition($label_key, $match, $match_operator);
    }
    if ($configuration['sort']['field'] !== '_none'
      && is_string($configuration['sort']['field'])
      && is_string($configuration['sort']['direction'])) {
      $query->sort($configuration['sort']['field'], $configuration['sort']['direction']);
    }

    return $query;
  }

}
