<?php

declare(strict_types=1);

namespace Drupal\refreshless\Cache\Context;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\Context\CalculatedCacheContextInterface;
use Drupal\refreshless\Service\RefreshlessKillSwitchInterface;
use function t;

/**
 * Defines the RefreshLess enabled cache context service.
 *
 * Cache context ID: 'refreshless_enabled'.
 *
 * This varies based on whether the current request has RefreshLess enabled;
 * i.e. whether the kill switch was triggered. Note that a request can be a
 * RefreshLess request but still result in a full page load response.
 */
class RefreshlessEnabledCacheContext implements CalculatedCacheContextInterface {

  /**
   * Constructor; saves dependencies.
   *
   * @param \Drupal\refreshless\Service\RefreshlessKillSwitchInterface $killSwitch
   *   The RefreshLess kill switch service.
   */
  public function __construct(
    protected readonly RefreshlessKillSwitchInterface $killSwitch,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getLabel() {
    return t('RefreshLess enabled');
  }

  /**
   * {@inheritdoc}
   */
  public function getContext($parameter = null) {

    return match ($this->killSwitch->triggered()) {
      true  => 'true',
      false => 'false',
    };

  }

  /**
   * {@inheritdoc}
   */
  public function getCacheableMetadata($parameter = null) {
    return new CacheableMetadata();
  }

}
