/**
 * @file
 * Fixes scroll sometimes not resetting to the top on forward navigation.
 *
 * This works around a Firefox issue where the browser will sometimes not
 * scroll to the top as expected on an advance visit (i.e. clicking a link,
 * not a back or forward history navigation) if scroll-behavior: smooth; is
 * set on the <html> element. The solution is to temporarily force
 * scroll-behavior: auto; (instant scrolling) right as the visit starts and
 * remove it once the load has occurred.
 *
 * Also note that Turbo at the time of writing seems to add aria-busy to
 * <html> on clicking an in-page anchor (such as a table of contents) and not
 * remove it until navigated to a different page. Because of this, we can't
 * use a CSS-only solution.
 *
 * @see https://github.com/hotwired/turbo/issues/190#issuecomment-1525135845
 *   CSS-only solution for future reference.
 *
 * @see https://github.com/hotwired/turbo/issues/426
 *
 * @see https://github.com/hotwired/turbo/issues/698
 */
(function(html, Drupal, $, once) {

  'use strict';

  /**
   * Our event namespace.
   *
   * @type {String}
   *
   * @see https://learn.jquery.com/events/event-basics/#namespacing-events
   */
  const eventNamespace = 'refreshless-turbo-disable-smooth-scroll';

  /**
   * The once() identifier for our behaviour.
   *
   * @type {String}
   */
  const onceName = eventNamespace;

  /**
   * Class added to the <html> element during a load to prevent smooth scroll.
   *
   * @type {String}
   */
  const smoothScrollDisableClass = 'refreshless-disable-smooth-scroll';

  const loadHandler = async (event) => {

    // Wait for at least a frame to be rendered to reduce the chance we might
    // remove the class too early and the browser ends up smooth scrolling.
    await new Promise(requestAnimationFrame);
    await new Promise(requestAnimationFrame);

    $(event.target).removeClass(smoothScrollDisableClass);

  };

  const visitHandler = (event) => {

    $(event.target)
    .addClass(smoothScrollDisableClass)
    .one(`refreshless:load.${eventNamespace}`, loadHandler);

  };

  $(once(onceName, html)).on(`turbo:visit.${eventNamespace}`, visitHandler);

})(document.documentElement, Drupal, jQuery, once);
