<?php

declare(strict_types=1);

namespace Drupal\Tests\refreshless_turbo\FunctionalJavascript;

use Drupal\Core\Url;
use Drupal\Tests\refreshless_turbo\FunctionalJavascript\TurboWebDriverTestBase;

/**
 * Event tests.
 *
 * @group refreshless
 *
 * @group refreshless_turbo
 */
class EventTest extends TurboWebDriverTestBase {

  /**
   * The event list HTML id attribute name.
   */
  protected const EVENT_LIST_ID = 'refreshless-turbo-event-test-events';

  /**
   * The event list item event type attribute name.
   */
  protected const EVENT_TYPE_ATTR = 'data-event-type';

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block', 'refreshless_turbo_event_test', 'system',
  ];

  /**
   * Asserts that the provided event sequence was triggered.
   *
   * This uses the HTML list generated by our test module's JavaScript to
   * determine the events triggered up to the point of calling this method.
   *
   * @param array $events
   *   An array of associative arrays; each associative array must contain a
   *   'type' key with a value being the expected event type, and optionally may
   *   contain a 'content' key with a value being the expected HTML content of
   *   that event.
   */
  protected function assertEventSequence(array $events): void {

    $this->assertSession()->elementExists(
      'css', '#' . self::EVENT_LIST_ID,
    );

    foreach ($events as $i => $event) {

      $eventSelector = '#' . self::EVENT_LIST_ID . ' :nth-child(' . (
        $i + 1
      ) . ')';

      $element = $this->assertSession()->elementAttributeExists(
        'css', $eventSelector, self::EVENT_TYPE_ATTR,
      );

      $actual = (string) $element->getAttribute(self::EVENT_TYPE_ATTR);

      $this->assertEquals($event['type'], $actual);

      if (empty($event['content'])) {
        continue;
      }

      $this->assertSession()->elementContains(
        'css',
        $eventSelector,
        $event['content'],
      );

    }

  }

  /**
   * Test that RefreshLess events are triggered in the expected order.
   */
  public function testEventSequence(): void {

    $this->drupalPlaceBlock('local_tasks_block', [
      'region' => 'content', 'id' => 'local-tasks-block',
    ]);

    $loginUrl = Url::fromRoute('user.login');

    $registerUrl = Url::fromRoute('user.register');

    $resetPasswordUrl = Url::fromRoute('user.pass');

    $this->drupalGet($loginUrl);

    $this->assertSession()->assertRefreshlessIsPresent();

    $this->assertSession()->startRefreshlessPersist();

    $this->click('[data-drupal-link-system-path="' .
      $registerUrl->getInternalPath() .
    '"]');

    $this->assertWaitOnRefreshlessRequest();

    $this->assertSession()->elementExists(
      'css', 'form[data-drupal-selector="user-register-form"]',
    );

    $this->click('[data-drupal-link-system-path="' .
      $resetPasswordUrl->getInternalPath() .
    '"]');

    $this->assertWaitOnRefreshlessRequest();

    $this->assertSession()->elementExists(
      'css', 'form[data-drupal-selector="user-pass"]',
    );

    $this->assertEventSequence([
      [
        'type'    => 'navigation-response',
        'content' => 'navigated to <code>' .
          $registerUrl->setAbsolute(false)->toString() .
        '</code>',
      ],
      [
        'type' => 'before-cache',
      ],
      [
        'type' => 'detach',
      ],
      [
        'type' => 'drupal-settings-update',
      ],
      [
        'type' => 'attach',
      ],
      [
        'type'    => 'navigation-response',
        'content' => 'navigated to <code>' .
          $resetPasswordUrl->setAbsolute(false)->toString() .
        '</code>',
      ],
      [
        'type' => 'before-cache',
      ],
      [
        'type' => 'detach',
      ],
      [
        'type' => 'drupal-settings-update',
      ],
      [
        'type' => 'attach',
      ],
    ]);

  }

}
