<?php

declare(strict_types=1);

namespace Drupal\refreshless\Value;

use Drupal\refreshless\Value\RequestWrapperInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Default implementation of a RefreshLess request wrapper value object.
 *
 * This is intended to be overridden by RefreshLess implementations and does
 * nothing otherwise.
 */
class RequestWrapper implements RequestWrapperInterface {

  /**
   * Name of the header indicating a prefetch follow up notify request.
   */
  protected const PREFETCH_NOTIFY_HEADER_NAME = 'X-RefreshLess-Prefetch-Notify';

  /**
   * Cookie name that disables RefreshLess for a request if found.
   *
   * Note that the value doesn't matter, just the presence of the cookie.
   */
  protected const DISABLE_COOKIE_NAME = 'refreshless-disable';

  /**
   * The request object this value object was constructed from, if any.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected readonly Request $request;

  /**
   * Protected constructor; use static contructor methods instead.
   */
  protected function __construct() {}

  /**
   * {@inheritdoc}
   */
  public function isRefreshless(): bool {
    return false;
  }

  /**
   * {@inheritdoc}
   */
  public function isReload(): bool {
    return false;
  }

  /**
   * {@inheritdoc}
   */
  public function getReloadReason(): string {
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public static function fromRequest(Request $request): self {

    // Note that static must be used and not self to allow instantiating classes
    // that extend this one automagically.
    $instance = new static();

    $instance->request = $request;

    return $instance;

  }

  /**
   * {@inheritdoc}
   */
  public function getRequest(): Request {
    return $this->request;
  }

  /**
   * {@inheritdoc}
   */
  public function isDisabled(): bool {
    return $this->request->cookies->has(self::DISABLE_COOKIE_NAME);
  }

  /**
   * Get the disable RefreshLess cookie name.
   *
   * @return string
   *   The disable RefreshLess cookie name.
   *
   * @internal This is primarily intended for internal use to allow a single
   *   source of truth for the cookie name.
   */
  public static function getDisableCookieName(): string {
    return self::DISABLE_COOKIE_NAME;
  }

  /**
   * {@inheritdoc}
   */
  public function isPrefetch(): bool {
    return false;
  }

  /**
   * {@inheritdoc}
   */
  public function isPrefetchNotify(): bool {
    return $this->request->headers->has(self::PREFETCH_NOTIFY_HEADER_NAME);
  }

  /**
   * Get the prefetch notify header name.
   *
   * @return string
   *   The prefetch notify header name.
   *
   * @internal This is primarily intended for internal use to allow a single
   *   source of truth for the header name.
   */
  public static function getPrefetchNotifyHeaderName(): string {
    return self::PREFETCH_NOTIFY_HEADER_NAME;
  }

  /**
   * {@inheritdoc}
   */
  public function isPreload(): bool {
    return false;
  }

}
