<?php

declare(strict_types=1);

namespace Drupal\refreshless_turbo\Value;

use Drupal\refreshless\Value\RequestWrapper as BaseRequestWrapper;
use Symfony\Component\HttpFoundation\Request;

/**
 * Hotwire Turbo implementation of a RefreshLess request wrapper value object.
 */
class RequestWrapper extends BaseRequestWrapper {

  /**
   * Name of the header indicating a RefreshLess request.
   */
  protected const REQUEST_HEADER_NAME = 'x-refreshless-turbo';

  /**
   * Name of the cookie passed by our JavaScript indicating a reload.
   */
  protected const RELOAD_REASON_COOKIE_NAME = 'refreshless-turbo-reload-reason';

  /**
   * {@inheritdoc}
   */
  public function isRefreshless(): bool {
    return $this->request->headers->has(self::REQUEST_HEADER_NAME);
  }

  /**
   * {@inheritdoc}
   */
  public function isReload(): bool {
    return $this->request->cookies->has(self::RELOAD_REASON_COOKIE_NAME);
  }

  /**
   * {@inheritdoc}
   */
  public function getReloadReason(): string {

    if (!$this->isReload()) {
      return '';
    }

    return $this->request->cookies->get(self::RELOAD_REASON_COOKIE_NAME);

  }

  /**
   * Get the reload reason cookie name.
   *
   * @return string
   *   The reload reason cookie name.
   *
   * @internal This is primarily intended for internal use to allow a single
   *   source of truth for the cookie name. External code should not have to
   *   know the cookie name, nor whether a cookie or some other method is used.
   */
  public static function getReloadReasonCookieName(): string {
    return self::RELOAD_REASON_COOKIE_NAME;
  }

  /**
   * Get the HTTP header name identifying a RefreshLess request.
   *
   * @return string
   *   The RefreshLess request HTTP header name.
   *
   * @internal This is primarily intended for internal use to allow a single
   *   source of truth for the header name. External code should not have to
   *   know the header name, nor whether a header or some other method is used.
   */
  public static function getRequestHeaderName(): string {
    return self::REQUEST_HEADER_NAME;
  }

  /**
   * {@inheritdoc}
   */
  public function isPrefetch(): bool {
    return $this->request->headers->get('X-Sec-Purpose', '') === 'prefetch';
  }

  /**
   * {@inheritdoc}
   */
  public function isPreload(): bool {
    return $this->request->headers->get('X-Sec-Purpose', '') === 'preload';
  }

}
