<?php

declare(strict_types=1);

namespace Drupal\refreshless\Cache\Context;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\Context\CalculatedCacheContextInterface;
use Drupal\refreshless\Service\RequestWrapperFactoryInterface;
use function t;

/**
 * Defines the RefreshLess request cache context service.
 *
 * Cache context ID: 'refreshless_request'.
 *
 * This varies based on whether the current request was performed through
 * RefreshLess. Note that this does not indicate whether the request will result
 * in a RefreshLess load or a full page load.
 */
class RefreshlessRequestCacheContext implements CalculatedCacheContextInterface {

  /**
   * Constructor; saves dependencies.
   *
   * @param \Drupal\refreshless\Service\RequestWrapperFactoryInterface $requestWrapperFactory
   *   The RefreshLess request wrapper factory.
   */
  public function __construct(
    protected readonly RequestWrapperFactoryInterface $requestWrapperFactory,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getLabel() {
    return t('RefreshLess request');
  }

  /**
   * {@inheritdoc}
   */
  public function getContext($parameter = null) {

    return match (
      $this->requestWrapperFactory->fromRequest()->isRefreshless()
    ) {
      true  => 'true',
      false => 'false',
    };

  }

  /**
   * {@inheritdoc}
   */
  public function getCacheableMetadata($parameter = null) {
    return new CacheableMetadata();
  }

}
