/**
 * @file
 * Drupal messages compatibility for RefreshLess.
 */
(function(html, $, once) {

  'use strict';

  /**
   * Our event namespace.
   *
   * @type {String}
   *
   * @see https://learn.jquery.com/events/event-basics/#namespacing-events
   */
  const eventNamespace = 'refreshless-turbo-messages-compatibility';

  /**
   * Our once() identifier.
   *
   * @type {String}
   */
  const onceName = eventNamespace;

  // Remove Drupal messages from previews before they're rendered.
  //
  // These are inherently temporary and in response to various actions that are
  // not usually repeated by navigating back to a page.
  //
  // Note that we're specifically looking for the messages placed after the
  // fallback element to avoid false positives where a message may be embedded
  // in content and thus is content that should be cached.
  //
  // Also note that we're doing this during 'refreshless:before-render' instead
  // of 'refreshless:before-cache' because the latter cannot be delayed while
  // the former can be. This is necessary to prevent messages instantly
  // disappearing before a page has transitioned out when transitions are
  // active, as those need to delay rendering.
  $(once(
    onceName, html,
  )).on(`refreshless:before-render.${eventNamespace}`, async (event) => {

    // Note that we only want to remove these from previews and not all cached
    // snapshots, so that when navigating via back/forward and no network
    // request is made (if cache heads indicate that's okay), messages are still
    // displayed like they would be if navigating with full page loads.
    if (event.detail.isPreview === false) {
      return;
    }

    // Delay rendering to ensure we can manipulate the new content before it's
    // rendered.
    await event.detail.delay(async (resolve, reject) => {

      $(event.detail.newBody).find([
        '[data-drupal-messages-fallback]',
        '[data-drupal-messages]',
      ].join(' + ')).remove();

      // Resume rendering now that we've altered the content.
      resolve();

    });

  });

})(document.documentElement, jQuery, once);
