<?php

declare(strict_types=1);

namespace Drupal\refreshless\Value;

use Drupal\refreshless\Value\ValueObjectWithRequestInterface;

/**
 * Interface for a RefreshLess request wrapper value object.
 */
interface RequestWrapperInterface extends ValueObjectWithRequestInterface {

  /**
   * Whether the wrapped request is a RefreshLess request.
   *
   * @return bool
   *   True if the request was sent via RefreshLess, false otherwise.
   */
  public function isRefreshless(): bool;

  /**
   * Whether the wrapped request is a RefreshLess reload request.
   *
   * A reload request is one that was requested by the front-end JavaScript but
   * will not be shown to the user, instead resulting in a full page reload.
   *
   * @return bool
   *   True if this is a reload request, false otherwise.
   *
   * @see self::getReloadReason()
   *   To get the reload reason.
   */
  public function isReload(): bool;

  /**
   * Get the reload reason, if any.
   *
   * @return string
   *   A string code indicating the reload reason, or an empty string if this is
   *   not a reload request.
   *
   * @see self::isReload()
   *   To determine if this is a reload request.
   */
  public function getReloadReason(): string;

  /**
   * Whether the wrapped request indicated RefreshLess should be disabled.
   *
   * @return bool
   *   True if the wrapped request indicated RefreshLess should be disabled,
   *   false otherwise.
   */
  public function isDisabled(): bool;

  /**
   * Whether this request is a prefetch request.
   *
   * Prefetch requests are usually in anticipation of a user interaction and
   * tend to occur lazily as a result. The prefetched response may or may be
   * used, depending on what the client decides, and may be discarded after a
   * certain amount of time if not displayed to the user.
   *
   * @return bool
   *   True if the wrapped request is a prefetch request, false otherwise.
   */
  public function isPrefetch(): bool;

  /**
   * Whether this request is a prefetch follow-up request to notify it was used.
   *
   * @return bool
   *   True if this was a prefetch follow-up, false otherwise.
   */
  public function isPrefetchNotify(): bool;

  /**
   * Whether this request is a preload request.
   *
   * A preload request is similar to a prefetch but has a different purpose:
   * preloads may be used for permanent client-side caching and are often
   * invoked programmatically rather than as a response to user interactions.
   * Like prefetches, they may or may not be displayed to the user.
   *
   * @return bool
   *   True if the wrapped request is a preload request, false otherwise.
   */
  public function isPreload(): bool;

}
