<?php

namespace Drupal\register_with_otp\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\RegisterForm;
use Exception;
use Symfony\Component\HttpFoundation\Session\SessionInterface;

/**
 * Extends the core register form to introduce otp functionality.
 */
class RegisterWithOtp extends RegisterForm {

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    parent::save($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state) {
    $element = parent::actions($form, $form_state);
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);
    $session = $this->getRequest()->getSession();
    if ($this->currentUser()->isAnonymous()) {
      $form['#attached']['library'][] = 'register_with_otp/verify_otp';
      $form['account']['mail']['#prefix'] = '<div id="email-field">';
      $form['account']['mail']['#suffix'] = '</div>';
      if (!$session->get('otp_verified')) {
        $form['actions']['send_otp'] = [
          '#prefix' => '<div id="send-otp-btn">',
          '#suffix' => '</div>',
          '#type' => 'submit', 
          '#value' => $this->t('Verify email'),
          '#attributes' => [
            'hidden' => $session->get('otp'),
          ],
          '#ajax' => [
            'callback' => '::sendOtp',
            'event' => 'click',
            'progress' => [
              'type' => 'throbber',
            ],
          ],
        ];
      }
      $form['container'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Verify OTP'),
        '#attributes' => [
          'id' => 'otp-form',
          'class' => ['form-card'],
          'hidden' => !$session->get('otp'),
        ],
      ];
      $form['container']['otp'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Enter your OTP'),
        '#required' => TRUE,
        '#pattern' => '[0-9]{5}',
        '#attributes' => [
          'id' => 'otp-field',
        ],
        '#maxlength' => 5,
        '#description' => $this->t('Please enter the numeric OTP sent to your email.'),
      ];
      
      $form['actions']['submit']['#attributes']['class'][] = 'create-account-btn';
      $form['actions']['submit']['#attributes']['hidden'] = !$session->get('otp');
    }
    return $form;
  }

  /**
   * Function to validate the email input and send otp to the email if correct.
   * 
   * @param array $form
   *   Takes the form build array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Takes the form state instance of the form.
   * 
   * @return void
   */
  public function sendOtp(&$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $session = $this->getRequest()->getSession();
    $email = trim($form_state->getValue('mail'));
    $message = '';
    $email = trim($form_state->getValue('mail'));
    $success = FALSE;

    $user = $this->entityTypeManager->getStorage('user')->loadByProperties([
      'mail' => $email,
    ]);
    if (!$email || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
      $message = 'Email address is invalid.';
      $form['account']['mail']['#suffix'] = '<div class="email-error">' . $message . '</div>';
      $response->addCommand(new ReplaceCommand('#email-field', $form['account']['mail']));
    }
    elseif ($user) {
      $form['account']['mail']['#suffix'] = '<div class="email-error">Email is already taken</div>';
      $response->addCommand(new ReplaceCommand('#email-field', $form['account']['mail']));
    }
    elseif ($session->get('otp_verified') && $email === $session->get('email')) {
      $message = $email . ' is already verified.';
      $form['account']['mail']['#suffix'] = '<div class="email-message">' . $message . '</div>';
      $response->addCommand(new ReplaceCommand('#email-field', $form['account']['mail']));
    }
    elseif ($session->get('otp') && $email === $session->get('email')) {
      if ((time() - $session->get('time')) <= 300) {
        $form['account']['mail']['#suffix'] = '</div>';
        $response->addCommand(new ReplaceCommand('#email-field', $form['account']['mail']));
      }
      else {
        $message = 'The OTP is expired. Kindly generate a new one to verify your email and create an account..';
        $form['account']['mail']['#suffix'] = '<div class="email-error">' . $message . '</div>';
        $form['actions']['send_otp']['#value'] = $this->t('Resend OTP');
        $response->addCommand(new ReplaceCommand('#email-field', $form['account']['mail']));
        $response->addCommand(new ReplaceCommand('#send-otp-btn', $form['actions']['send_otp']));
        $this->resetUserSession($form, $form_state);
      }
    }
    else {
      // Sending the otp after email syntax verification.
      $otp = random_int(10000, 90000);
      $email = $form_state->getValue('mail');
      $data['subject'] = 'OTP veriification';
      $data['content'] = ['otp' => $otp];
      $timestamp = time();
      
      $message = $this->sendOtpMail($otp, $data, $email, $timestamp, $session);

      if ($message) {
        $success = TRUE;
      }
      if ($success) {
        $form['account']['mail']['#suffix'] = '</div>';
        $form['container']['#attributes']['hidden'] = FALSE;
        $form['actions']['send_otp']['#attributes']['hidden'] = TRUE;
        $form['actions']['submit']['#attributes']['hidden'] = FALSE;
        $response->addCommand(new ReplaceCommand('.create-account-btn', $form['actions']['submit']));
        $response->addCommand(new ReplaceCommand('#send-otp-btn', $form['actions']['send_otp']));
        $response->addCommand(new ReplaceCommand('#otp-form', $form['container']));
        $response->addCommand(new ReplaceCommand('#email-field', $form['account']['mail']));
      }
      else {
        $form['account']['mail']['#suffix'] = '<div class="email-error">' . $message . '</div>';
        $response->addCommand(new ReplaceCommand('#email-field', $form['account']['mail']));
      }
    }

    $this->messenger()->deleteAll();
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
    
    $session = $this->getRequest()->getSession();
    if ($this->currentUser()->isAnonymous()) {
      if ($session->get('otp') && $session->get('email') != $form_state->getValue('mail')) {
        $form_state->setErrorByName('mail', $this->t("You can't change your email after OTP verification, kindly reverify"));
        $user = $this->entityTypeManager->getStorage('user')->loadByProperties([
          'mail' => $form_state->getValue('mail'),
        ]);
        if (!$user) {
          /** Resending the mail when the email is changed before or after
           * verification. */
          $otp = random_int(10000, 90000);
          $email = $form_state->getValue('mail');
          $data['subject'] = 'OTP veriification';
          $data['content'] = ['otp' => $otp];
          $timestamp = time();
          $message = $this->sendOtpMail($otp, $data, $email, $timestamp, $session);

          if ($message) {
            $this->messenger()->addMessage($message);
          }
          else{
            $form_state->setErrorByName('mail', $this->t("Failed to send mail to given email address"));
            $session->set('otp_verified', FALSE);
            return;
          }
        }
        else {
          return;
        }
      }

      if ((time() - $session->get('time')) <= 300) {
        if ($form_state->getValue('otp') != $session->get('otp')) {
          $form_state->setErrorByName('otp', $this->t("Invalid OTP Entered"));
          $session->set('otp_verified', FALSE);
          return;
        }
        else {
          $session->set('otp_verified', TRUE);
        }
      }
      else {
        $this->resetUserSession($form, $form_state);

        $form_state->setErrorByName('otp', $this->t("The OTP is expired, kindly get a new otp to verify."));
        $form['actions']['submit']['#attributes']['hidden'] = !$session->get('otp');
        $form['actions']['send_otp']['#attributes']['hidden'] = $session->get('otp');
        $form['container']['#attributes']['hidden'] = !$session->get('otp');

        $form_state->setRebuild(TRUE);
        return;
      }

      if (!$session->get('otp_verified')) {
        $form_state->setErrorByName('mail', $this->t("Please verify your email first."));
        $session->set('otp_verified', FALSE);
        return;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    if ($this->currentUser()->isAnonymous()) {
      $this->resetUserSession($form, $form_state);
    }
    parent::submitForm($form, $form_state);
  }

  /**
   * Function to reset the otp data stored in Session.
   * 
   * @param array $form
   *   Takes the form render array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Takes the form state instance.
   * 
   * @return void
   */
  protected function resetUserSession(array &$form, FormStateInterface $form_state) {
    $session = $this->getRequest()->getSession();
    $session->remove('otp');
    $session->remove('time');
    $session->remove('email');
    $session->remove('otp_verified');
  }

  /**
   * Function to send the otp in mail.
   * 
   * @param int $otp
   *   Takes the otp to be mailed to the given email.
   * @param array $data
   *   Takes the data array to be send in mail.
   * @param string $email
   *   Takes the email to be send mail upon.
   * @param int $timestamp
   *   Takes the current timestamp.
   * @param \Symfony\Component\HttpFoundation\Session\SessionInterface $session
   *   Takes the session instance to be manipulated.
   * 
   * @return string|bool
   *   Returns the success message or FALSE upon failure.
   */
  protected function sendOtpMail(int $otp, array $data, string $email, int $timestamp, SessionInterface $session) {
    $mailManager = \Drupal::service('plugin.manager.mail');
    try {
      $mailManager->mail("register_with_otp", "otp_mail_verification", $email, 'en', $data, NULL, TRUE);
      $session->set('otp', $otp);
      $session->set('time', $timestamp);
      $session->set('otp_verified', FALSE);
      $session->set('email', $email);
      $message = 'An OTP has been sent to your new email ' . $email;
      $this->logger('register_with_otp')->notice($this->t("A new OTP requested: ") . $otp);
      $this->logger('register_with_otp')->notice($this->t("A new OTP requested: ") . $otp);

      return $message;
    }
    catch (Exception $e) {
      $this->logger('register_with_otp')->notice($e->getMessage());
      return FALSE;
    }
  }

}
