<?php

namespace Drupal\registered_organisations;

/**
 * A profile class for structuring organisation information.
 */
class OrganisationProfile implements OrganisationProfileInterface {

  /**
   * The register plugin where this company was found.
   *
   * @var string
   */
  protected string $register = '';

  /**
   * The organisation id / number.
   *
   * @var string
   */
  protected string $id = '';

  /**
   * The organisation name.
   *
   * @var string
   */
  protected string $name = '';

  /**
   * The organisation type.
   *
   * @var string
   */
  protected string $type = '';

  /**
   * The organisation status.
   *
   * @var string
   */
  protected string $status = '';

  /**
   * The organisation's industry classification.
   *
   * @var array
   */
  protected array $industryClassificationCodes = [];
  /**
   * The organisation's industry classification system, one of SIC, NACE, NAICS.
   *
   * @var string
   */
  protected string $industryClassificationSystem = '';

  /**
   * The raw organisation data.
   *
   * @var array
   */
  protected array $data = [];

  /**
   * Constructs a \Drupal\Component\Plugin\PluginBase object.
   *
   * @param array $data
   *   An array of data.
   */
  public function __construct(array $data) {
    $this->data = $data;

    /*
     * Populate the data, an array with the following keys is expected:
     *
     * [
     *   'register': string,
     *   'id': string,
     *   'name': string,
     *   'type': string,
     *   'status': string,
     *   'classification': array ['codes': array, 'system': string]
     * ]
     */
    $this->hydrate($data);
  }

  /**
   * Hydrate the object by passing data from an array.
   */
  public function hydrate($data) {
    foreach ($data as $prop => $elem) {
      $method_name = 'set' . ucwords(strtolower($prop));
      if (!empty($elem) && method_exists($this, $method_name)) {
        $this->{$method_name}($elem);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  public function getRegister(): string {
    return $this->register;
  }

  /**
   * {@inheritDoc}
   */
  public function getId(): string {
    return $this->id;
  }

  /**
   * {@inheritDoc}
   */
  public function getName(): string {
    return $this->name;
  }

  /**
   * {@inheritDoc}
   */
  public function getType(): string {
    return $this->type;
  }

  /**
   * {@inheritDoc}
   */
  public function getStatus(): string {
    return $this->status;
  }

  /**
   * {@inheritDoc}
   */
  public function getClassification(): array|null {
    return (!empty($this->industryClassificationCodes) && !empty($this->industryClassificationSystem)) ? [
      'codes' => $this->industryClassificationCodes,
      'system' => $this->industryClassificationSystem,
    ] : NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function getData(): array {
    return $this->data;
  }

  /**
   * Setter for the self::$register property.
   *
   * @param string $plugin
   *   The plugin this organisation is retrievable from.
   */
  public function setRegister(string $plugin) {
    $this->register = (string) $plugin;
  }

  /**
   * Setter for the self::$id property.
   *
   * @param string $id
   *   The organisation number.
   */
  public function setId(string $id) {
    $this->id = (string) $id;
  }

  /**
   * Setter for the self::$name property.
   *
   * @param string $name
   *   The name of the organisation.
   */
  public function setName(string $name) {
    $this->name = (string) $name;
  }

  /**
   * Setter for the self::$type property.
   *
   * @param string $type
   *   The type of the organisation.
   */
  public function setType(string $type) {
    $this->type = (string) $type;
  }

  /**
   * Setter for the self::$status property.
   *
   * @param string $status
   *   The status of the organisation.
   */
  public function setStatus(string $status) {
    $this->status = (string) $status;
  }

  /**
   * Setter for the classification properties.
   *
   * @param array $classification
   *   The classification data (optional), should be an
   *   array ['codes': array, 'system': string].
   */
  public function setClassification(array $classification) {
    if (!empty($classification['codes']) && !empty($classification['system'])) {
      $this->industryClassificationCodes = $classification['codes'];
      $this->industryClassificationSystem = $classification['system'];
    }
  }

}
