<?php

namespace Drupal\registered_organisations\Plugin\OrganisationRegister;

use Drupal\registered_organisations\OrganisationProfile;
use Drupal\registered_organisations\OrganisationProfileInterface;
use Drupal\registered_organisations\OrganisationRegisterApi;
use Drupal\registered_organisations\OrganisationRegisterInterface;
use UKGovernmentBEIS\CompaniesHouse\ApiException;
use UKGovernmentBEIS\CompaniesHouse\Client as CompaniesClient;
use UKGovernmentBEIS\CompaniesHouse\NotFoundException;
use UKGovernmentBEIS\CompaniesHouse\RateLimitException;
use UKGovernmentBEIS\CompaniesHouse\UnauthorisedException;

/**
 * Cease a member.
 *
 * @OrganisationRegister(
 *   id = "companies_house",
 *   title = @Translation("UK companies register through Companies House.")
 * )
 */
class CompaniesHouseRegister extends OrganisationRegisterApi {

  /**
   * The company type enum.
   *
   * @see https://github.com/companieshouse/api-enumerations/blob/master/constants.yml
   */
  const COMPANY_TYPE = [
    'private-unlimited' => "Private unlimited company",
    'ltd' => "Private limited company",
    'plc' => "Public limited company",
    'old-public-company' => "Old public company",
    'private-limited-guarant-nsc-limited-exemption' => "Private Limited Company by guarantee without share capital, use of 'Limited' exemption",
    'limited-partnership' => "Limited partnership",
    'private-limited-guarant-nsc' => "Private limited by guarantee without share capital",
    'converted-or-closed' => "Converted / closed",
    'private-unlimited-nsc' => "Private unlimited company without share capital",
    'private-limited-shares-section-30-exemption' => "Private Limited Company, use of 'Limited' exemption",
    'protected-cell-company' => "Protected cell company",
    'assurance-company' => "Assurance company",
    'oversea-company' => "Overseas company",
    'eeig-establishment' => "European Economic Interest Grouping Establishment (EEIG)",
    'icvc-securities' => "Investment company with variable capital",
    'icvc-warrant' => "Investment company with variable capital",
    'icvc-umbrella' => "Investment company with variable capital",
    'registered-society-non-jurisdictional' => "Registered society",
    'industrial-and-provident-society' => "Industrial and Provident society",
    'northern-ireland' => "Northern Ireland company",
    'northern-ireland-other' => "Credit union (Northern Ireland)",
    'llp' => "Limited liability partnership",
    'royal-charter' => "Royal charter company",
    'investment-company-with-variable-capital' => "Investment company with variable capital",
    'unregistered-company' => "Unregistered company",
    'other' => "Other company type",
    'european-public-limited-liability-company-se' => "European public limited liability company (SE)",
    'united-kingdom-societas' => "United Kingdom Societas",
    'uk-establishment' => "UK establishment company",
    'scottish-partnership' => "Scottish qualifying partnership",
    'charitable-incorporated-organisation' => "Charitable incorporated organisation",
    'scottish-charitable-incorporated-organisation' => "Scottish charitable incorporated organisation",
    'further-education-or-sixth-form-college-corporation' => "Further education or sixth form college corporation",
    'eeig' => "European Economic Interest Grouping (EEIG)",
    'ukeig' => "United Kingdom Economic Interest Grouping",
    'registered-overseas-entity' => "Overseas entity",
  ];

  /**
   * The company status enum.
   *
   * @see https://github.com/companieshouse/api-enumerations/blob/master/constants.yml
   */
  const COMPANY_STATUS = [
    'active' => "Active",
    'dissolved' => "Dissolved",
    'liquidation' => "Liquidation",
    'receivership' => "Receiver Action",
    'converted-closed' => "Converted / Closed",
    'voluntary-arrangement' => "Voluntary Arrangement",
    'insolvency-proceedings' => "Insolvency Proceedings",
    'administration' => "In Administration",
    'open' => "Open",
    'closed' => "Closed",
    'registered' => "Registered",
    'removed' => "Removed",
  ];

  /**
   * {@inheritDoc}
   */
  public function getClient() {
    $config = \Drupal::config('registered_organisations.settings');

    $api_key = $config->get('companies_house_api_key');
    if (empty($api_key)) {
      return;
    }

    try {
      return new CompaniesClient($api_key);
    }
    catch (\Exception $e) {
      $this->getLogger()->warning("Failed to create Companies House Client using API: @message",
        ['@message' => $e->getMessage()]);
    }
  }

  /**
   * {@inheritdoc}
   */
  private function handleException(\Throwable $e) {
    switch (TRUE) {
      case $e instanceof NotFoundException:
        $this->getLogger()->info("The resource does not exist: @message",
          ['@message' => $e->getMessage()]);
        break;

      case $e instanceof RateLimitException:
        // Record the fact that rate limiting has been reached.
        $cid = implode(':', [$this->getCachePrefix(), 'rate_limit']);
        $expiry = $this->getRequestTime() + OrganisationRegisterInterface::RATE_LIMIT_TIMEOUT;
        $this->getCache()->set($cid, $this->getRequestTime(), $expiry);

        $this->getLogger()->warning("Too many requests, rate limit reached.");
        break;

      case $e instanceof UnauthorisedException:
        $this->getLogger()->error("Unauthorised access of the resource: @message",
          ['@message' => $e->getMessage()]);
        break;

      case $e instanceof ApiException:
        $this->getLogger()->error("Failed to get the resource: @message",
          ['@message' => $e->getMessage()]);
        break;

      default:
        $this->getLogger()->error("API connection error: @message",
          ['@message' => $e->getMessage()]);
    }
  }

  /**
   * {@inheritDoc}
   */
  public function getCompany(string $id): OrganisationProfileInterface|null {
    $company = &drupal_static(__FUNCTION__ . ':' . $id);
    if (isset($company)) {
      return $company;
    }

    // If rate limiting has been hit wait for 5 minutes to try requests again.
    $cid = implode(':', [$this->getCachePrefix(), 'rate_limit']);
    if ($this->getCache()->get($cid)) {
      return NULL;
    }

    try {
      $company_information = $this->client?->companyProfile($id);
    }
    catch (\Throwable $e) {
      $this->handleException($e);
      return NULL;
    }

    // Set the mandatory properties.
    $company_information += [
      'register' => $this->getPluginId(),
      'id' => $company_information['company_number'],
      'name' => $company_information['company_name'],
      'status' => $company_information['company_status'],
      'classification' => [
        'codes' => $company_information['sic_codes'],
        'system' => 'SIC',
      ],
    ];
    return new OrganisationProfile($company_information);
  }

  /**
   * {@inheritDoc}
   */
  public function findCompany(string $name): array {
    $companies = &drupal_static(__FUNCTION__ . ':' . $name);
    if (isset($companies)) {
      return $companies;
    }

    // If rate limiting has been hit wait for 5 minutes to try requests again.
    $cid = implode(':', [$this->getCachePrefix(), 'rate_limit']);
    if ($this->getCache()->get($cid)) {
      return [];
    }

    try {
      $company_search = $this->client?->searchCompanies($name);
    }
    catch (\Throwable $e) {
      $this->handleException($e);
      return [];
    }

    // Set the mandatory properties.
    $companies = [];
    foreach ($company_search["items"] as $search_item) {
      $search_item += [
        'register' => $this->getPluginId(),
        'id' => $search_item['company_number'],
        'name' => $search_item['company_name'],
        'status' => $search_item['company_status'],
      ];
      $companies[$search_item['company_number']] = new OrganisationProfile($search_item);
    }
    return $companies;
  }

}
